<?php

namespace Square\Traits;

use Square\Types\CatalogCustomAttributeValue;
use Square\Types\CatalogV1Id;
use Square\Core\Json\JsonProperty;
use Square\Core\Types\ArrayType;

/**
 * @property string $id
 * @property ?string $updatedAt
 * @property ?int $version
 * @property ?bool $isDeleted
 * @property ?array<string, CatalogCustomAttributeValue> $customAttributeValues
 * @property ?array<CatalogV1Id> $catalogV1Ids
 * @property ?bool $presentAtAllLocations
 * @property ?array<string> $presentAtLocationIds
 * @property ?array<string> $absentAtLocationIds
 * @property ?string $imageId
 */
trait CatalogObjectBase
{
    /**
     * An identifier to reference this object in the catalog. When a new `CatalogObject`
     * is inserted, the client should set the id to a temporary identifier starting with
     * a "`#`" character. Other objects being inserted or updated within the same request
     * may use this identifier to refer to the new object.
     *
     * When the server receives the new object, it will supply a unique identifier that
     * replaces the temporary identifier for all future references.
     *
     * @var string $id
     */
    #[JsonProperty('id')]
    private string $id;

    /**
     * Last modification [timestamp](https://developer.squareup.com/docs/build-basics/working-with-dates) in RFC 3339 format, e.g., `"2016-08-15T23:59:33.123Z"`
     * would indicate the UTC time (denoted by `Z`) of August 15, 2016 at 23:59:33 and 123 milliseconds.
     *
     * @var ?string $updatedAt
     */
    #[JsonProperty('updated_at')]
    private ?string $updatedAt;

    /**
     * The version of the object. When updating an object, the version supplied
     * must match the version in the database, otherwise the write will be rejected as conflicting.
     *
     * @var ?int $version
     */
    #[JsonProperty('version')]
    private ?int $version;

    /**
     * If `true`, the object has been deleted from the database. Must be `false` for new objects
     * being inserted. When deleted, the `updated_at` field will equal the deletion time.
     *
     * @var ?bool $isDeleted
     */
    #[JsonProperty('is_deleted')]
    private ?bool $isDeleted;

    /**
     * A map (key-value pairs) of application-defined custom attribute values. The value of a key-value pair
     * is a [CatalogCustomAttributeValue](entity:CatalogCustomAttributeValue) object. The key is the `key` attribute
     * value defined in the associated [CatalogCustomAttributeDefinition](entity:CatalogCustomAttributeDefinition)
     * object defined by the application making the request.
     *
     * If the `CatalogCustomAttributeDefinition` object is
     * defined by another application, the `CatalogCustomAttributeDefinition`'s key attribute value is prefixed by
     * the defining application ID. For example, if the `CatalogCustomAttributeDefinition` has a `key` attribute of
     * `"cocoa_brand"` and the defining application ID is `"abcd1234"`, the key in the map is `"abcd1234:cocoa_brand"`
     * if the application making the request is different from the application defining the custom attribute definition.
     * Otherwise, the key used in the map is simply `"cocoa_brand"`.
     *
     * Application-defined custom attributes are set at a global (location-independent) level.
     * Custom attribute values are intended to store additional information about a catalog object
     * or associations with an entity in another system. Do not use custom attributes
     * to store any sensitive information (personally identifiable information, card details, etc.).
     *
     * @var ?array<string, CatalogCustomAttributeValue> $customAttributeValues
     */
    #[JsonProperty('custom_attribute_values'), ArrayType(['string' => CatalogCustomAttributeValue::class])]
    private ?array $customAttributeValues;

    /**
     * The Connect v1 IDs for this object at each location where it is present, where they
     * differ from the object's Connect V2 ID. The field will only be present for objects that
     * have been created or modified by legacy APIs.
     *
     * @var ?array<CatalogV1Id> $catalogV1Ids
     */
    #[JsonProperty('catalog_v1_ids'), ArrayType([CatalogV1Id::class])]
    private ?array $catalogV1Ids;

    /**
     * If `true`, this object is present at all locations (including future locations), except where specified in
     * the `absent_at_location_ids` field. If `false`, this object is not present at any locations (including future locations),
     * except where specified in the `present_at_location_ids` field. If not specified, defaults to `true`.
     *
     * @var ?bool $presentAtAllLocations
     */
    #[JsonProperty('present_at_all_locations')]
    private ?bool $presentAtAllLocations;

    /**
     * A list of locations where the object is present, even if `present_at_all_locations` is `false`.
     * This can include locations that are deactivated.
     *
     * @var ?array<string> $presentAtLocationIds
     */
    #[JsonProperty('present_at_location_ids'), ArrayType(['string'])]
    private ?array $presentAtLocationIds;

    /**
     * A list of locations where the object is not present, even if `present_at_all_locations` is `true`.
     * This can include locations that are deactivated.
     *
     * @var ?array<string> $absentAtLocationIds
     */
    #[JsonProperty('absent_at_location_ids'), ArrayType(['string'])]
    private ?array $absentAtLocationIds;

    /**
     * @var ?string $imageId Identifies the `CatalogImage` attached to this `CatalogObject`.
     */
    #[JsonProperty('image_id')]
    private ?string $imageId;

    /**
     * @return string
     */
    public function getId(): string
    {
        return $this->id;
    }

    /**
     * @param string $value
     */
    public function setId(string $value): self
    {
        $this->id = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getUpdatedAt(): ?string
    {
        return $this->updatedAt;
    }

    /**
     * @param ?string $value
     */
    public function setUpdatedAt(?string $value = null): self
    {
        $this->updatedAt = $value;
        return $this;
    }

    /**
     * @return ?int
     */
    public function getVersion(): ?int
    {
        return $this->version;
    }

    /**
     * @param ?int $value
     */
    public function setVersion(?int $value = null): self
    {
        $this->version = $value;
        return $this;
    }

    /**
     * @return ?bool
     */
    public function getIsDeleted(): ?bool
    {
        return $this->isDeleted;
    }

    /**
     * @param ?bool $value
     */
    public function setIsDeleted(?bool $value = null): self
    {
        $this->isDeleted = $value;
        return $this;
    }

    /**
     * @return ?array<string, CatalogCustomAttributeValue>
     */
    public function getCustomAttributeValues(): ?array
    {
        return $this->customAttributeValues;
    }

    /**
     * @param ?array<string, CatalogCustomAttributeValue> $value
     */
    public function setCustomAttributeValues(?array $value = null): self
    {
        $this->customAttributeValues = $value;
        return $this;
    }

    /**
     * @return ?array<CatalogV1Id>
     */
    public function getCatalogV1Ids(): ?array
    {
        return $this->catalogV1Ids;
    }

    /**
     * @param ?array<CatalogV1Id> $value
     */
    public function setCatalogV1Ids(?array $value = null): self
    {
        $this->catalogV1Ids = $value;
        return $this;
    }

    /**
     * @return ?bool
     */
    public function getPresentAtAllLocations(): ?bool
    {
        return $this->presentAtAllLocations;
    }

    /**
     * @param ?bool $value
     */
    public function setPresentAtAllLocations(?bool $value = null): self
    {
        $this->presentAtAllLocations = $value;
        return $this;
    }

    /**
     * @return ?array<string>
     */
    public function getPresentAtLocationIds(): ?array
    {
        return $this->presentAtLocationIds;
    }

    /**
     * @param ?array<string> $value
     */
    public function setPresentAtLocationIds(?array $value = null): self
    {
        $this->presentAtLocationIds = $value;
        return $this;
    }

    /**
     * @return ?array<string>
     */
    public function getAbsentAtLocationIds(): ?array
    {
        return $this->absentAtLocationIds;
    }

    /**
     * @param ?array<string> $value
     */
    public function setAbsentAtLocationIds(?array $value = null): self
    {
        $this->absentAtLocationIds = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getImageId(): ?string
    {
        return $this->imageId;
    }

    /**
     * @param ?string $value
     */
    public function setImageId(?string $value = null): self
    {
        $this->imageId = $value;
        return $this;
    }
}
