<?php

namespace Square\Cards\Requests;

use Square\Core\Json\JsonSerializableType;
use Square\Core\Json\JsonProperty;
use Square\Types\Card;

class CreateCardRequest extends JsonSerializableType
{
    /**
     * A unique string that identifies this CreateCard request. Keys can be
     * any valid string and must be unique for every request.
     *
     * Max: 45 characters
     *
     * See [Idempotency keys](https://developer.squareup.com/docs/build-basics/common-api-patterns/idempotency) for more information.
     *
     * @var string $idempotencyKey
     */
    #[JsonProperty('idempotency_key')]
    private string $idempotencyKey;

    /**
     * @var string $sourceId The ID of the source which represents the card information to be stored. This can be a card nonce or a payment id.
     */
    #[JsonProperty('source_id')]
    private string $sourceId;

    /**
     * An identifying token generated by [Payments.verifyBuyer()](https://developer.squareup.com/reference/sdks/web/payments/objects/Payments#Payments.verifyBuyer).
     * Verification tokens encapsulate customer device information and 3-D Secure
     * challenge results to indicate that Square has verified the buyer identity.
     *
     * See the [SCA Overview](https://developer.squareup.com/docs/sca-overview).
     *
     * @var ?string $verificationToken
     */
    #[JsonProperty('verification_token')]
    private ?string $verificationToken;

    /**
     * @var Card $card Payment details associated with the card to be stored.
     */
    #[JsonProperty('card')]
    private Card $card;

    /**
     * @param array{
     *   idempotencyKey: string,
     *   sourceId: string,
     *   card: Card,
     *   verificationToken?: ?string,
     * } $values
     */
    public function __construct(
        array $values,
    ) {
        $this->idempotencyKey = $values['idempotencyKey'];
        $this->sourceId = $values['sourceId'];
        $this->verificationToken = $values['verificationToken'] ?? null;
        $this->card = $values['card'];
    }

    /**
     * @return string
     */
    public function getIdempotencyKey(): string
    {
        return $this->idempotencyKey;
    }

    /**
     * @param string $value
     */
    public function setIdempotencyKey(string $value): self
    {
        $this->idempotencyKey = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function getSourceId(): string
    {
        return $this->sourceId;
    }

    /**
     * @param string $value
     */
    public function setSourceId(string $value): self
    {
        $this->sourceId = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getVerificationToken(): ?string
    {
        return $this->verificationToken;
    }

    /**
     * @param ?string $value
     */
    public function setVerificationToken(?string $value = null): self
    {
        $this->verificationToken = $value;
        return $this;
    }

    /**
     * @return Card
     */
    public function getCard(): Card
    {
        return $this->card;
    }

    /**
     * @param Card $value
     */
    public function setCard(Card $value): self
    {
        $this->card = $value;
        return $this;
    }
}
