<?php

namespace f12_cf7_captcha\core\timer;

use f12_cf7_captcha\core\wpdb;
use Forge12\Shared\LoggerInterface;
use RuntimeException;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Captcha Timer
 * Generate the custom captcha as an image
 *
 * @package forge12\contactform7
 */
class CaptchaTimer {

	/**
	 * The unique ID
	 *
	 * @var int
	 */
	private $id = 0;
	/**
	 * The identifier used in the contact form
	 *
	 * @var string
	 */
	private $hash = '';
	/**
	 * The value - stores the time in milliseconds
	 *
	 * @var float|string
	 */
	private $value = '';
	/**
	 * The datetime whenever the captcha code has been created
	 *
	 * @var string
	 */
	private $createtime = '';

	/**
	 * @var string
	 */
	private $updatetime = '';
	private LoggerInterface $logger;

	/**
	 * Create a new Captcha Object
	 *
	 * @param $object
	 */
	public function __construct(LoggerInterface $logger, $params = array())
	{
		$this->logger = $logger;

		$this->get_logger()->info('Konstruktor gestartet.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
		]);

		$this->set_params($params);

		$this->get_logger()->debug('Initialisierung mit Parametern abgeschlossen.', [
			'params_count' => count($params),
		]);
	}

	private function get_logger(): LoggerInterface {
		return $this->logger;
	}

	/**
	 * Set the parameters for the object.
	 *
	 * @param array $params An associative array of parameter values.
	 *                      The keys of the array should correspond to the property names of the object.
	 *                      The values of the array should be the new values for the corresponding properties.
	 *
	 * @return void
	 */
	private function set_params(array $params)
	{
		$this->get_logger()->info('Versuche, Parameter auf die Klasseneigenschaften anzuwenden.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'given_params_count' => count($params),
		]);

		foreach ($params as $key => $value) {
			if (isset($this->{$key})) {
				$this->{$key} = $value;
				$this->get_logger()->debug("Parameter '{$key}' wurde erfolgreich gesetzt.", [
					'value' => $value,
				]);
			} else {
				$this->get_logger()->warning("Parameter '{$key}' existiert nicht als Klasseneigenschaft. Er wird übersprungen.");
			}
		}

		$this->get_logger()->info('Parameter-Initialisierung abgeschlossen.');
	}

	/**
	 * Retrieves the table name for the CF7 Captcha Timer plugin.
	 *
	 * This method returns the name of the table that is used by the CF7 Captcha Timer plugin for storing data.
	 *
	 * @return string The table name for the CF7 Captcha Timer plugin.
	 *
	 * @throws RuntimeException When the global $wpdb is not defined.
	 */
	public function get_table_name(): string
	{
		global $wpdb;

		$this->get_logger()->info('Versuche, den Namen der Timer-Datenbanktabelle abzurufen.');

		if (null === $wpdb) {
			$error_message = 'Die globale Variable $wpdb ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		$table_name = $wpdb->prefix . 'f12_cf7_captcha_timer';

		$this->get_logger()->debug('Tabellenname erfolgreich abgerufen.', [
			'table_name' => $table_name,
		]);

		return $table_name;
	}

	/**
	 * Creates a table in the database.
	 *
	 * This method uses the WordPress function dbDelta() to create a table with the specified structure.
	 * The table name is generated by calling the get_table_name() method.
	 *
	 * @return void
	 */
	public function create_table(): void
	{
		$this->get_logger()->info('Starte die Erstellung der Datenbanktabelle.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
		]);

		// Lade die WordPress-Upgrade-Funktionalität.
		require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
		$this->get_logger()->debug('Die WordPress-Upgrade-Funktionalität wurde geladen.');

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Tabellenerstellung abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			return;
		}

		global $wpdb;

		$sql = sprintf( "CREATE TABLE %s (
                id int(11) NOT NULL auto_increment, 
                hash varchar(255) NOT NULL, 
                value varchar(255) NOT NULL,
                createtime varchar(255) DEFAULT '',
                PRIMARY KEY  (id)
            )", $table_name );

		$this->get_logger()->debug('SQL-Anweisung zur Tabellenerstellung vorbereitet.', [
			'sql' => $sql,
		]);

		// Führe die dbDelta-Funktion aus.
		dbDelta($sql);

		// Es gibt keine Methode "table_exists" in der Klasse.
		// Eine Überprüfung des Erfolgs kann nicht protokolliert werden.
		$this->get_logger()->info('Tabellenerstellung abgeschlossen.');
	}

	/**
	 * Delete the specified table from the database
	 *
	 * @throws RuntimeException if WPDB is not defined
	 */
	public function delete_table(): void
	{
		$this->get_logger()->info('Starte den Prozess zum Löschen der Datenbanktabelle und des Cron-Jobs.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
		]);

		global $wpdb;

		if (null === $wpdb) {
			$error_message = 'WPDB ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Löschvorgang abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			return;
		}

		// Lösche die Datenbanktabelle
		$sql = sprintf("DROP TABLE IF EXISTS %s", $table_name);
		$wpdb->query($sql);
		$this->get_logger()->info('Datenbanktabelle wurde gelöscht (falls vorhanden).', ['table_name' => $table_name]);

		// Lösche den geplanten Cron-Job
		wp_clear_scheduled_hook('dailyCaptchaTimerClear');
		$this->get_logger()->info('Geplanter Cron-Job "dailyCaptchaTimerClear" wurde gelöscht.');

		$this->get_logger()->info('Löschvorgang der Tabelle und des Cron-Jobs erfolgreich abgeschlossen.');
	}


	/**
	 * Get the ID of the object
	 *
	 * @return int The ID of the object
	 */
	public function get_id(): int
	{
		$this->get_logger()->debug('Rufe die ID des Objekts ab.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'id' => $this->id,
		]);

		return $this->id;
	}

	/**
	 * Set the id of the object
	 *
	 * @param int $id The id to be set
	 *
	 * @return void
	 */
	public function set_id(int $id): void
	{
		$this->get_logger()->info('Setze die ID des Objekts.', [
			'class'  => __CLASS__,
			'method' => __METHOD__,
			'old_id' => $this->id,
			'new_id' => $id,
		]);

		$this->id = $id;

		$this->get_logger()->debug('ID erfolgreich auf ' . $id . ' gesetzt.');
	}


	/**
	 * Retrieves the hash value generated for this object.
	 * If the hash value is not yet generated, it calls the generateHash method to generate one and stores it.
	 *
	 * @return string The hash value.
	 */
	public function get_hash(string $user_ip_address = ''): string
	{
		$this->get_logger()->info('Versuche, den Hash-Wert abzurufen.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'current_hash_state' => empty($this->hash) ? 'leer' : 'vorhanden',
		]);

		// Generiere einen neuen Hash, wenn das aktuelle Objekt keinen hat
		if (empty($this->hash)) {
			$this->get_logger()->debug('Aktueller Hash-Wert ist leer. Generiere einen neuen.');
			$this->hash = $this->generate_hash($user_ip_address);
			$this->get_logger()->info('Neuer Hash-Wert wurde generiert.');
		} else {
			$this->get_logger()->debug('Hash-Wert ist bereits vorhanden. Gebe den bestehenden Wert zurück.');
		}

		return $this->hash;
	}

	/**
	 * Generates a hash string using the user's IP address concatenated with the current time.
	 *
	 * @param string $user_ip_address The IP address of the user.
	 *
	 * @return string The generated hash string.
	 */
	private function generate_hash(string $user_ip_address): string
	{
		$this->get_logger()->info('Generiere einen neuen eindeutigen Hash-Wert.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
		]);

		// Generiere eine einzigartige, schwer zu erratende Zeichenkette
		$data_to_hash = microtime(true) . $user_ip_address . uniqid('', true);

		// Erstelle den Hash unter Verwendung von password_hash zur Stärkung der Sicherheit
		$hash = password_hash($data_to_hash, PASSWORD_DEFAULT);

		$this->get_logger()->debug('Hash-Generierung abgeschlossen.', [
			'generated_hash_length' => strlen($hash),
		]);

		return $hash;
	}

	/**
	 * Checks if the hash is valid.
	 *
	 * @return bool Returns true if the hash is valid, false otherwise.
	 */
	private function is_valid_hash(): bool
	{
		$is_valid = !empty($this->hash);

		$this->get_logger()->debug('Überprüfe die Gültigkeit des Hash-Werts.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'hash_state' => $is_valid ? 'gültig' : 'ungültig',
		]);

		return $is_valid;
	}


	/**
	 * Retrieves the value of the property.
	 *
	 * @TODO Check if that ever returns anything else than a float. - should be the time in ms for timer validation
	 *
	 * @return string|float The value of the property as a string.
	 */
	public function get_value() {
		return $this->value;
	}

	/**
	 * Sets the value of the property.
	 *
	 * @TODO Check if that ever returns anything else than a float. - should be the time in ms for timer validation
	 *
	 * @param $value The value to be set.
	 *
	 * @return void
	 */
	public function set_value($value)
	{
		$this->get_logger()->info('Setze den Wert des Objekts.', [
			'class'  => __CLASS__,
			'method' => __METHOD__,
			'old_value' => $this->value,
			'new_value' => $value,
		]);

		$this->value = $value;

		$this->get_logger()->debug('Wert erfolgreich gesetzt.');
	}

	/**
	 * Returns the create time of the object.
	 *
	 * If the create time is not set, it will be generated using the current
	 * date and time.
	 *
	 * @return string The create time in the format 'Y-m-d H:i:s'
	 */
	public function get_create_time(): string
	{
		$this->get_logger()->info('Versuche, die Erstellungszeit abzurufen.');

		// Überprüfe, ob die Erstellungszeit bereits gesetzt ist
		if (empty($this->createtime)) {
			$this->get_logger()->debug('Erstellungszeit ist leer. Generiere einen neuen Zeitstempel.');

			try {
				// Erstelle ein neues DateTime-Objekt mit der aktuellen Zeit
				$dt = new \DateTime();
				$this->createtime = $dt->format('Y-m-d H:i:s');
				$this->get_logger()->info('Neuer Zeitstempel erfolgreich generiert.', ['createtime' => $this->createtime]);
			} catch (\Exception $e) {
				$this->get_logger()->error('Fehler beim Erstellen des DateTime-Objekts.', ['error' => $e->getMessage()]);
				// Optional: Rückgabe eines Standard-Zeitstempels bei Fehler
				return date('Y-m-d H:i:s');
			}
		} else {
			$this->get_logger()->debug('Erstellungszeit ist bereits vorhanden. Gebe den bestehenden Wert zurück.');
		}

		return $this->createtime;
	}

	/**
	 * Returns the create time of the object.
	 *
	 * If the create time is not set, it will be generated using the current
	 * date and time.
	 *
	 * @return string The create time in the format 'Y-m-d H:i:s'
	 */
	public function get_update_time(): string
	{
		$this->get_logger()->info('Versuche, die Aktualisierungszeit abzurufen.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
		]);

		// Überprüfe, ob die Aktualisierungszeit bereits gesetzt ist.
		if (empty($this->updatetime)) {
			$this->get_logger()->debug('Aktualisierungszeit ist leer. Generiere einen neuen Zeitstempel.');

			try {
				// Erstelle ein neues DateTime-Objekt mit der aktuellen Zeit.
				$dt = new \DateTime();
				$this->updatetime = $dt->format('Y-m-d H:i:s');
				$this->get_logger()->info('Neuer Zeitstempel erfolgreich generiert.', ['updatetime' => $this->updatetime]);
			} catch (\Exception $e) {
				$this->get_logger()->error('Fehler beim Erstellen des DateTime-Objekts.', ['error' => $e->getMessage()]);
				// Optional: Rückgabe eines Standard-Zeitstempels bei Fehler.
				return date('Y-m-d H:i:s');
			}
		} else {
			$this->get_logger()->debug('Aktualisierungszeit ist bereits vorhanden. Gebe den bestehenden Wert zurück.');
		}

		return $this->updatetime;
	}

	/**
	 * Sets the creation time of the object.
	 *
	 * This function sets the creation time of the object to the current date and time in the format 'Y-m-d H:i:s'.
	 * It updates the 'updatetime' property with the current date and time.
	 *
	 * @return void
	 */
	private function set_create_time(): void
	{
		$this->get_logger()->info('Setze die Erstellungszeit für das Objekt.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
		]);

		try {
			$dt = new \DateTime();
			$this->createtime = $dt->format('Y-m-d H:i:s');
			$this->get_logger()->debug('Erstellungszeit erfolgreich gesetzt.', ['createtime' => $this->createtime]);
		} catch (\Exception $e) {
			$this->get_logger()->error('Fehler beim Erstellen des DateTime-Objekts. Erstellungszeit konnte nicht gesetzt werden.', [
				'error' => $e->getMessage(),
			]);
		}
	}

	/**
	 * Retrieves a CaptchaTimer object by its ID.
	 *
	 * @param int $id The ID of the CaptchaTimer to retrieve.
	 *
	 * @return CaptchaTimer|null The retrieved CaptchaTimer object, or null if not found.
	 * @throws RuntimeException If WPDB global variable is not defined.
	 */
	public function get_by_id(int $id): ?CaptchaTimer
	{
		$this->get_logger()->info('Versuche, einen Captcha-Timer nach ID abzurufen.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'id' => $id,
		]);

		global $wpdb;

		if (!$wpdb) {
			$error_message = 'Die globale Variable $wpdb ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Abfrage abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			return null;
		}

		// Verwende eine sichere Methode, um SQL-Injection zu verhindern.
		// Die WordPress-Funktion prepare ist dafür ideal.
		$sql = $wpdb->prepare("SELECT * FROM {$table_name} WHERE id = %d", $id);
		$this->get_logger()->debug('SQL-Abfrage vorbereitet.', ['sql' => $sql]);

		$results = $wpdb->get_results($sql, ARRAY_A);

		if (empty($results)) {
			$this->get_logger()->info('Kein Captcha-Timer-Eintrag für die gegebene ID gefunden.', ['id' => $id]);
			return null;
		}

		// Erstelle ein CaptchaTimer-Objekt aus den ersten Ergebnissen
		$timer = new CaptchaTimer($this->get_logger(), $results[0]);

		$this->get_logger()->info('Captcha-Timer-Eintrag erfolgreich nach ID abgerufen.', [
			'id' => $timer->get_id(),
			'hash' => $timer->get_hash(),
		]);

		return $timer;
	}

	/**
	 * Retrieves the CaptchaTimer object from the database based on the given hash.
	 *
	 * @param string $hash The hash value associated with the CaptchaTimer object.
	 *
	 * @return CaptchaTimer|null The retrieved CaptchaTimer object if found, or null if not found.
	 *
	 * @throws RuntimeException When the global $wpdb is not defined.
	 */
	public function get_by_hash(string $hash): ?CaptchaTimer
	{
		$this->get_logger()->info('Versuche, einen Captcha-Timer nach Hash abzurufen.', [
			'class'  => __CLASS__,
			'method' => __METHOD__,
			'hash'   => $hash,
		]);

		global $wpdb;

		if (!$wpdb) {
			$error_message = 'Die globale Variable $wpdb ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Abfrage abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			return null;
		}

		// Verwende $wpdb->prepare() zur SQL-Injection-Prävention.
		$sql = $wpdb->prepare("SELECT * FROM {$table_name} WHERE hash = %s", $hash);
		$this->get_logger()->debug('SQL-Abfrage vorbereitet.', ['sql' => $sql]);

		$results = $wpdb->get_results($sql, ARRAY_A);

		if (empty($results)) {
			$this->get_logger()->info('Kein Captcha-Timer-Eintrag für den gegebenen Hash gefunden.', ['hash' => $hash]);
			return null;
		}

		// Da wir wissen, dass das Array nicht leer ist, können wir den ersten Eintrag verwenden.
		$timer = new CaptchaTimer($this->get_logger(), $results[0]);
		$this->get_logger()->info('Captcha-Timer-Eintrag erfolgreich nach Hash abgerufen.', [
			'id' => $timer->get_id(),
			'hash' => $timer->get_hash(),
		]);

		return $timer;
	}

	/**
	 * Deletes the object by its hash.
	 *
	 * @return bool Returns `true` if the deletion is successful, `false` otherwise.
	 */
	public function delete(): bool
	{
		$this->get_logger()->info('Versuche, den aktuellen Timer-Eintrag zu löschen.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'hash' => $this->hash,
		]);

		if (empty($this->hash)) {
			$this->get_logger()->warning('Kein Hash-Wert zum Löschen vorhanden. Vorgang abgebrochen.');
			return false;
		}

		$is_deleted = $this->delete_by_hash($this->hash);

		if ($is_deleted) {
			$this->get_logger()->info('Timer-Eintrag erfolgreich gelöscht.');
			// Optional: Setze den internen Hash auf null, um Mehrfachlöschungen zu verhindern.
			$this->hash = null;
		} else {
			$this->get_logger()->error('Fehler beim Löschen des Timer-Eintrags.');
		}

		return $is_deleted;
	}


	/**
	 * Deletes a record from the database table based on the given hash.
	 *
	 * @param string $hash The hash of the record to be deleted.
	 *
	 * @return bool True if the record was successfully deleted, false otherwise.
	 * @throws RuntimeException If WPDB global variable is not defined.
	 */
	public function delete_by_hash(string $hash): bool
	{
		$this->get_logger()->info('Starte den Löschvorgang basierend auf dem Hash-Wert.', [
			'class'  => __CLASS__,
			'method' => __METHOD__,
			'hash'   => $hash,
		]);

		global $wpdb;

		if (!$wpdb) {
			$error_message = 'Die globale Variable $wpdb ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Löschvorgang abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			return false;
		}

		// Verwende $wpdb->prepare() zur SQL-Injection-Prävention.
		$sql = $wpdb->prepare("DELETE FROM {$table_name} WHERE hash = %s", $hash);
		$this->get_logger()->debug('SQL-Löschabfrage vorbereitet.', ['sql' => $sql]);

		// Führe die Abfrage aus. query() gibt die Anzahl der betroffenen Zeilen oder false bei einem Fehler zurück.
		$result = $wpdb->query($sql);

		if ($result === false) {
			$this->get_logger()->error('Fehler beim Ausführen der Löschabfrage.', [
				'db_error' => $wpdb->last_error,
			]);
			return false;
		}

		$this->get_logger()->info('Löschvorgang erfolgreich abgeschlossen.', [
			'rows_deleted' => $result,
			'hash' => $hash,
		]);

		// Rückgabe des Ergebnisses der Abfrage. True, wenn eine oder mehrere Zeilen gelöscht wurden.
		return (bool)$result;
	}


	/**
	 * Checks if the current instance of the class is an update operation.
	 *
	 * @return bool Returns true if the instance is an update operation, otherwise false.
	 */
	private function is_update(): bool
	{
		$is_update = ($this->id != 0);

		$this->get_logger()->debug('Überprüfe, ob das Objekt aktualisiert oder neu erstellt werden soll.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'current_id' => $this->id,
			'is_update' => $is_update,
		]);

		return $is_update;
	}

	/**
	 * Reset the table in the database.
	 *
	 * This method deletes all rows from the table specified by the get_table_name() method.
	 * It uses the global $wpdb object for executing the SQL query.
	 *
	 * @return int The number of rows affected by the delete operation.
	 * @throws RuntimeException If $wpdb is not defined.
	 *
	 */
	public function reset_table(): int
	{
		$this->get_logger()->info('Starte den Vorgang, um die gesamte Tabelle zu leeren.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
		]);

		global $wpdb;

		if (!$wpdb) {
			$error_message = 'Die globale Variable $wpdb ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Reset abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			return 0;
		}

		// Verwende sprintf zur Vorbereitung der Abfrage, da hier keine Benutzereingaben verarbeitet werden.
		$sql = sprintf('TRUNCATE TABLE %s', $table_name);
		// TRUNCATE ist in der Regel schneller und effizienter als DELETE ohne WHERE-Klausel.
		// Jedoch gibt es keine Rückgabe der gelöschten Zeilenanzahl.

		// Alternativ, um die Anzahl der gelöschten Zeilen zu erhalten:
		$sql = sprintf('DELETE FROM %s', $table_name);

		$this->get_logger()->debug('SQL-Abfrage zum Zurücksetzen der Tabelle vorbereitet.', ['sql' => $sql]);

		// Führe die Abfrage aus und erhalte die Anzahl der betroffenen Zeilen.
		$rows_deleted = $wpdb->query($sql);

		if ($rows_deleted === false) {
			$this->get_logger()->error('Fehler beim Ausführen der Abfrage zum Zurücksetzen der Tabelle.', [
				'db_error' => $wpdb->last_error,
			]);
			return 0;
		}

		$this->get_logger()->info('Tabelle erfolgreich zurückgesetzt.', [
			'rows_deleted' => $rows_deleted,
		]);

		return $rows_deleted;
	}

	/**
	 * Deletes records from the database that are older than the specified create time.
	 *
	 * @param string $create_time The create time to compare against.
	 *
	 * @return int Returns the number of affected rows after the delete operation.
	 *
	 * @throws RuntimeException When WPDB is not defined.
	 */
	public function delete_older_than(string $create_time): int
	{
		$this->get_logger()->info('Starte den Löschvorgang für Einträge, die älter als ein bestimmter Zeitstempel sind.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'create_time' => $create_time,
		]);

		global $wpdb;

		if (!$wpdb) {
			$error_message = 'Die globale Variable $wpdb ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Löschvorgang abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			return 0;
		}

		// Verwende $wpdb->prepare() zur SQL-Injection-Prävention.
		// Der Zeitstempel sollte als string behandelt werden (%s).
		$sql = $wpdb->prepare("DELETE FROM {$table_name} WHERE createtime < %s", $create_time);
		$this->get_logger()->debug('SQL-Löschabfrage vorbereitet.', ['sql' => $sql]);

		// Führe die Abfrage aus und erhalte die Anzahl der betroffenen Zeilen.
		$rows_deleted = $wpdb->query($sql);

		if ($rows_deleted === false) {
			$this->get_logger()->error('Fehler beim Ausführen der Löschabfrage.', [
				'db_error' => $wpdb->last_error,
			]);
			return 0;
		}

		$this->get_logger()->info('Alte Timer-Einträge erfolgreich gelöscht.', [
			'rows_deleted' => $rows_deleted,
		]);

		return (int)$rows_deleted;
	}

	/**
	 * Retrieves the number of entries in the specified table.
	 *
	 * @return int The number of entries in the table.
	 * @throws RuntimeException If WPDB is not defined.
	 */
	public function get_count(): int
	{
		$this->get_logger()->info('Versuche, die Gesamtzahl der Einträge in der Tabelle abzurufen.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
		]);

		global $wpdb;

		if (!$wpdb) {
			$error_message = 'Die globale Variable $wpdb ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Abfrage abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			return 0;
		}

		// Verwende sprintf für die Abfrage, da hier keine Benutzereingaben verwendet werden.
		$sql = sprintf('SELECT count(*) AS entries FROM %s', $table_name);
		$this->get_logger()->debug('SQL-Abfrage vorbereitet.', ['sql' => $sql]);

		$results = $wpdb->get_results($sql);

		if (is_array($results) && isset($results[0])) {
			$count = (int)$results[0]->entries;
			$this->get_logger()->info("Anzahl der Einträge in der Tabelle: {$count}.");
			return $count;
		}

		$this->get_logger()->warning('Ergebnis der Datenbankabfrage war ungültig oder leer. Rückgabe von 0.');

		return 0;
	}

	/**
	 * Save the current object to the database.
	 *
	 * If the object's ID is not set, a new row will be inserted into the database.
	 * If the object's ID is set, the corresponding row in the database will be updated.
	 *
	 * @return int|bool The number of rows affected on successful update, or the ID of the inserted row on
	 *                  successful insert. False on failure.
	 * @throws RuntimeException If wpdb is not defined.
	 *
	 * @global wpdb $wpdb WordPress database object.
	 *
	 */
	public function save()
	{
		$this->get_logger()->info('Starte den Speichervorgang für den Timer-Eintrag.', [
			'class' => __CLASS__,
			'method' => __METHOD__,
			'action' => $this->is_update() ? 'update' : 'insert',
		]);

		global $wpdb;

		if (!$wpdb) {
			$error_message = 'Die globale Variable $wpdb ist nicht definiert.';
			$this->get_logger()->error($error_message);
			throw new RuntimeException($error_message);
		}

		try {
			$table_name = $this->get_table_name();
		} catch (RuntimeException $e) {
			$this->get_logger()->error('Fehler beim Abrufen des Tabellennamens. Speichervorgang abgebrochen.', [
				'error' => $e->getMessage(),
			]);
			throw $e;
		}

		$data = [
			'hash'       => $this->get_hash(),
			'value'      => $this->get_value(),
			'createtime' => $this->get_create_time(),
		];

		$result = false;
		if ($this->is_update()) {
			$where = ['id' => $this->get_id()];
			$this->get_logger()->debug('Aktualisiere bestehenden Eintrag.', ['data' => $data, 'where' => $where]);
			$result = $wpdb->update($table_name, $data, $where);
		} else {
			$this->get_logger()->debug('Füge neuen Eintrag hinzu.', ['data' => $data]);
			$result = $wpdb->insert($table_name, $data);
			if ($result !== false) {
				$this->id = $wpdb->insert_id;
				$this->get_logger()->info('Neuer Eintrag erfolgreich hinzugefügt. ID: ' . $this->id);
			}
		}

		if ($result === false) {
			$this->get_logger()->error('Datenbankfehler beim Speichern des Eintrags.', [
				'db_error' => $wpdb->last_error,
			]);
			throw new RuntimeException('Datenbankfehler aufgetreten. Bitte aktivieren Sie das Plugin erneut, um fehlende Tabellen zu erstellen.');
		}

		$this->get_logger()->info('Speichervorgang erfolgreich abgeschlossen.', [
			'result' => $result,
		]);

		return $result;
	}
}