<?php

namespace f12_cf7_captcha\core\protection\captcha;

use f12_cf7_captcha\CF7Captcha;
use f12_cf7_captcha\core\protection\javascript\JavascriptValidator;
use f12_cf7_captcha\core\TemplateController;
use f12_cf7_captcha\core\UserData;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CaptchaImageGenerator
 * Generate the custom captcha as an image
 *
 * @package forge12\contactform7
 */
class CaptchaImageGenerator extends CaptchaGenerator {
	/**
	 * @var string The font used for the captcha
	 */
	private $_font = '';

	/**
	 * If the image is going to be stored, we will reuse it
	 *
	 * @var string|null
	 */
	private ?string $image = null;

	/**
	 * Whether to save the image or not
	 *
	 * @var bool true or false, default: false
	 */
	private bool $store_image = false;

	/**
	 * constructor.
	 */
	public function __construct( CF7Captcha $Controller, int $length = 6 ) {
		parent::__construct( $Controller, $length );

		$this->_font = plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'assets/arial.ttf';

		if ( file_exists( $this->_font ) ) {
			$this->get_logger()->info(
				"__construct(): Captcha-Image-Generator initialisiert – Font geladen",
				[
					'plugin' => 'f12-cf7-captcha',
					'class'  => __CLASS__,
					'font'   => $this->_font
				]
			);
		} else {
			$this->get_logger()->error(
				"__construct(): Captcha-Image-Generator initialisiert – Font nicht gefunden",
				[
					'plugin' => 'f12-cf7-captcha',
					'class'  => __CLASS__,
					'font'   => $this->_font
				]
			);
		}
	}

	/**
	 * Enables or disables the image storing feature.
	 *
	 * @param bool $store_image [Optional] Whether to enable or disable the image storing. Default is true.
	 *
	 * @return void
	 */
	public function enable_image_storing( bool $store_image = true ): void {
		$this->store_image = $store_image;

		$this->get_logger()->info(
			"enable_image_storing(): Bildspeicherung umgeschaltet",
			[
				'plugin' => 'f12-cf7-captcha',
				'class'  => __CLASS__,
				'status' => $store_image ? 'aktiviert' : 'deaktiviert'
			]
		);
	}


	/**
	 * Retrieves the AJAX response generated by the current object.
	 *
	 * @return string The AJAX response generated by the current object.
	 */
	public function get_ajax_response(): string {
		$image = $this->get_image();

		if ( empty( $image ) ) {
			$this->get_logger()->error(
				"get_ajax_response(): Kein Captcha-Bild erzeugt",
				[
					'plugin' => 'f12-cf7-captcha',
					'class'  => __CLASS__
				]
			);

			return '';
		}

		$this->get_logger()->info(
			"get_ajax_response(): Captcha-Bild erfolgreich erzeugt",
			[
				'plugin'  => 'f12-cf7-captcha',
				'class'   => __CLASS__,
				'length'  => strlen( $image ),
				'preview' => substr( $image, 0, 30 ) . '...' // nur kleine Vorschau loggen, kein Klartext-Bild
			]
		);

		return $image;
	}

	/**
	 * Generates and returns the captcha image.
	 *
	 * @return string The captcha image HTML markup.
	 */
	public function get_image(): string {
		// the captcha
		$captcha = $this->get();

		if ( empty( $captcha ) ) {
			$this->get_logger()->error(
				"get_image(): Kein Captcha vorhanden → Bild konnte nicht generiert werden",
				[
					'plugin' => 'f12-cf7-captcha',
					'class'  => __CLASS__
				]
			);

			return '';
		}

		$this->get_logger()->debug(
			"get_image(): Starte Generierung neues Captcha-Bild",
			[
				'plugin'         => 'f12-cf7-captcha',
				'class'          => __CLASS__,
				'captcha_length' => strlen( $captcha )
			]
		);

		// Create the image
		$image = imagecreate( 125, 30 );
		imagecolorallocate( $image, 255, 255, 255 );

		// Positioning
		$offsetLeft = 10;

		for ( $i = 0; $i < strlen( $captcha ); $i ++ ) {
			imagettftext(
				$image,
				20,
				rand( - 10, 10 ),
				$offsetLeft + ( ( $i == 0 ? 5 : 15 ) * $i ),
				25,
				imagecolorallocate( $image, 200, 200, 200 ),
				$this->_font,
				$captcha[ $i ]
			);
			imagettftext(
				$image,
				16,
				rand( - 15, 15 ),
				$offsetLeft + ( ( $i == 0 ? 5 : 15 ) * $i ),
				25,
				imagecolorallocate( $image, 69, 103, 137 ),
				$this->_font,
				$captcha[ $i ]
			);
		}

		ob_start();
		imagepng( $image );
		$image = ob_get_contents();
		ob_end_clean();
		$rand = uniqid('cpi_', true);
		$this->image = '<span class="captcha-image"><img id="'.$rand.'" alt="captcha" src="data:image/png;base64,' . base64_encode( $image ) . '"/></span>';

		$this->get_logger()->info(
			"get_image(): Neues Captcha-Bild generiert",
			[
				'plugin' => 'f12-cf7-captcha',
				'class'  => __CLASS__,
				'bytes'  => strlen( $image )
			]
		);

		return $this->image;
	}


	/**
	 * @param string $captcha_code
	 * @param string $captcha_hash
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function is_valid( string $captcha_code, string $captcha_hash ): bool {
		/** @var UserData $User_Data */
		$User_Data  = $this->Controller->get_modul( 'user-data' );
		$ip_address = $User_Data->get_ip_address();

		$this->get_logger()->debug(
			"is_valid(): Starte Validierung",
			[
				'plugin' => 'f12-cf7-captcha',
				'ip'     => $ip_address,
				'hash'   => $captcha_hash
			]
		);

		$Captcha = new Captcha( $this->Controller->get_logger(), $ip_address );
		$Captcha = $Captcha->get_by_hash( $captcha_hash );

		if ( ! $Captcha ) {
			$this->get_logger()->warning(
				"is_valid(): Kein Captcha für Hash gefunden",
				[
					'plugin' => 'f12-cf7-captcha',
					'ip'     => $ip_address,
					'hash'   => substr( $captcha_hash, 0, 6 ) . '...'
				]
			);

			return false;
		}

		if ( $Captcha->get_validated() == 1 ) {
			$this->get_logger()->info(
				"is_valid(): Captcha bereits validiert → ungültig",
				[
					'plugin' => 'f12-cf7-captcha',
					'id'     => $Captcha->get_id()
				]
			);

			return false;
		}

		$Captcha->set_validated( 1 );
		$Captcha->save();

		if ( $captcha_code !== $Captcha->get_code() ) {
			$this->get_logger()->warning(
				"is_valid(): Code stimmt nicht überein → ungültig",
				[
					'plugin'    => 'f12-cf7-captcha',
					'id'        => $Captcha->get_id(),
					'submitted' => $captcha_code,
					'expected'  => $Captcha->get_code()
				]
			);

			return false;
		}

		$this->get_logger()->info(
			"is_valid(): Captcha erfolgreich validiert",
			[
				'plugin' => 'f12-cf7-captcha',
				'id'     => $Captcha->get_id()
			]
		);

		return true;
	}


	/**
	 * Retrieves the form field for a given field name.
	 *
	 * @param string         $field_name         The name of the field.
	 *
	 * @formatter:off
     *
     *  @param array  $args{
     *       An associative array of additional arguments:
     *
     *       @type string    $classes            The CSS classes for the captcha.
     *       @type string    $wrapper_classes    The CSS classes for the captcha wrapper.
     *       @type array     $attrbiutes         An associative array of additional HTML attributes for
     *                                           the captcha input field.
     *       @type array     $wrapper_attributes An associative array of additional HTML
     *                                           attributes for the captcha wrapper. Default values are provided for all
     *                                           arguments.
     *  }
     *
	 * @return string The HTML content of the form field or null if an error occurred.
	 *
	 * @formatter:on
	 *
	 *
	 * @since     1.0.0
	 */
	public function get_field( string $field_name, array $args = [] ): string {
		/*
		 * Parse the args
		 */
		$atts = [
			'classes'            => '',
			'wrapper_classes'    => '',
			'attributes'         => [],
			'wrapper_attributes' => [],
		];

		$atts = array_merge( $atts, $args );

		/**
		 * @var UserData $User_Data
		 */
		$User_Data  = $this->Controller->get_modul( 'user-data' );
		$ip_address = $User_Data->get_ip_address();

		/*
		 * Maybe generate the captcha session
		 */
		if ( $this->Captcha_Session != null ) {
			$Captcha_Session = $this->Captcha_Session;
			$this->get_logger()->debug(
				"get_field(): Vorhandene Captcha-Session wiederverwendet",
				[
					'plugin' => 'f12-cf7-captcha',
					'ip'     => $ip_address,
				]
			);
		} else {
			$Captcha_Session = new Captcha( $this->Controller->get_logger(), $ip_address );
			$this->get_logger()->debug(
				"get_field(): Neue Captcha-Session erstellt",
				[
					'plugin' => 'f12-cf7-captcha',
					'ip'     => $ip_address,
				]
			);
		}

		/*
		 * Update the captcha session values
		 */
		$Captcha_Session->set_code( $this->get() );
		$Captcha_Session->save();

		/*
		 * Store the session as latest session
		 */
		$this->Captcha_Session = $Captcha_Session;

		/*
		 * Get the label
		 */
		$label = $this->Controller->get_settings( 'protection_captcha_label', 'global' );

		/*
		 * Set the label
		 */
		#$label = sprintf("<div class=\"c-header\"><div class=\"c-label\">%s</div><div class=\"c-data\">%s</div><div class=\"c-reload\">%s</div></div>", $label, $this->get_image(), $this->get_reload_button());

		/*
		 * Parse the attributes
		 */
		$attributes = '';

		foreach ( $atts['attributes'] as $key => $value ) {
			$attributes .= esc_attr( $key ) . '="' . esc_attr( $value ) . '" ';
		}

		/*
		 * Parse the wrapper attributes
		 */
		$wrapper_attributes = '';
		foreach ( $atts['wrapper_attributes'] as $key => $value ) {
			$wrapper_attributes .= esc_attr( $key ) . '="' . esc_attr( $value ) . '" ';
		}

		$hash = $Captcha_Session->get_hash();

		/*
		 * Generate a unique ID
		 */
		$hash_id    = $this->get_last_unique_id_hash();
		$captcha_id = $this->get_last_unique_id_captcha();

		/*
		 * Get the placeholder
		 */
		$placeholder = $this->Controller->get_settings( 'protection_captcha_placeholder', 'global' );

		/*
		 * Generate the captcha html output
		 */
		#$captcha = sprintf('<input type="hidden" id="%s" name="%s_hash" value="%s"/>', esc_attr($hash_id), esc_attr($field_name), esc_attr($hash));

		if ( ! empty( $atts['classes'] ) ) {
			$atts['classes'] = ' ' . $atts['classes'];
		}

		/**
		 * Generate the captcha html output
		 *
		 * @var TemplateController $TemplateController
		 */
		$TemplateController = $this->Controller->get_modul( 'template' );


		/*
		 * Get Template
		 */
		$template = (int) $this->Controller->get_settings( 'protection_captcha_template', 'global' );

		if ( ! in_array( $template, [ 0, 1, 2 ], true ) ) {
			$template = 0;
		}

		$this->get_logger()->info(
			"get_field(): Captcha-Feld wird generiert",
			[
				'plugin'     => 'f12-cf7-captcha',
				'field_name' => $field_name,
				'template'   => $template,
				'hash_id'    => $hash_id,
				'captcha_id' => $captcha_id,
			]
		);

		$captcha = $TemplateController->get_plugin_template( 'captcha/template-' . $template, [
			'hash_id'            => $hash_id,
			'hash_field_name'    => $field_name . '_hash',
			'hash_value'         => $hash,
			'wrapper_classes'    => $atts['wrapper_classes'],
			'wrapper_attributes' => $wrapper_attributes,
			'label'              => $label,
			'classes'            => $atts['classes'],
			'attributes'         => $attributes,
			'captcha_id'         => $captcha_id,
			'field_name'         => $field_name,
			'placeholder'        => $placeholder,
			'captcha_data'       => $this->get_image(),
			'captcha_reload'     => $this->get_reload_button(),
			'method'             => 'image',
		] );


		#$captcha .= sprintf('<div class="%s" %s><label> %s</label><input class="f12c%s" data-method="image" %s type="text" id="%s" name="%s" placeholder="%s" value=""/></div>', $atts['wrapper_classes'], $wrapper_attributes, $label, $atts['classes'], $attributes, esc_attr($captcha_id), esc_attr($field_name), esc_attr($placeholder));

		/**
		 * Update the Math Field before output
		 *
		 * The filter allows developers to customize the form field for the honeypot before returning.
		 *
		 * @param string  $captcha         The HTML content of the form input field used as honeypot.
		 * @param string  $field_name      The Name of the field used as id and name for the input.
		 * @param string  $label           The Label for the captcha
		 * @param Captcha $Captcha_Session The Captcha Session storing the Captcha Information
		 * @param string  $classes         The CSS classes for the captcha
		 *
		 * @since 1.0.0
		 */
		$filtered = apply_filters( 'f12-cf7-captcha-get-form-field-image', $captcha, $field_name, $label, $Captcha_Session, $atts['classes'] );


		if ( $filtered !== $captcha ) {
			$this->get_logger()->debug(
				"get_field(): Captcha-Feld durch Filter modifiziert",
				[
					'plugin'     => 'f12-cf7-captcha',
					'field_name' => $field_name
				]
			);
		}

		return $filtered;
	}
}
