<?php
/**
 * Cron service
 *
 * @author hideokamoto <hide.okamoto@digitalcube.jp>
 * @since 6.1.1
 * @package C3_CloudFront_Cache_Controller
 */

namespace C3_CloudFront_Cache_Controller;
use C3_CloudFront_Cache_Controller\Constants;
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Cron service
 *
 * @since 6.1.1
 * @package C3_CloudFront_Cache_Controller
 */
class Cron_Service {
	/**
	 * Hook
	 *
	 * @var WP\Hooks
	 */
	private $hook_service;

	/**
	 * WP Transient service
	 *
	 * @var WP\Transient_Service
	 */
	private $transient_service;

	/**
	 * Log cron register task flag
	 *
	 * @var boolean
	 */
	private $log_cron_register_task;

	/**
	 * CloudFront service
	 *
	 * @var \C3_CloudFront_Cache_Controller\AWS\CloudFront_Service
	 */
	private $cf_service;

	/**
	 * Inject a external services
	 *
	 * @param mixed ...$args Inject class.
	 */
	function __construct( ...$args ) {
		$this->hook_service      = new WP\Hooks();
		$this->transient_service = new WP\Transient_Service();
		$this->cf_service        = new AWS\CloudFront_Service();

		if ( $args && ! empty( $args ) ) {
			foreach ( $args as $key => $value ) {
				if ( $value instanceof WP\Hooks ) {
					$this->hook_service = $value;
				} elseif ( $value instanceof WP\Transient_Service ) {
					$this->transient_service = $value;
				} elseif ( $value instanceof AWS\CloudFront_Service ) {
					$this->cf_service = $value;
				}
			}
		}
		$this->hook_service->add_action(
			'c3_cron_invalidation',
			array(
				$this,
				'run_schedule_invalidate',
			)
		);
		$this->log_cron_register_task = $this->hook_service->apply_filters( 'c3_log_cron_invalidation_task', $this->get_debug_setting( Constants::DEBUG_LOG_CRON_REGISTER_TASK ) );
	}

	/**
	 * Run the schedule invalidation
	 *
	 * @return boolean
	 */
	public function run_schedule_invalidate() {
		if ( $this->log_cron_register_task ) {
			error_log( '===== C3 Invalidation cron is started ===' );
		}
		if ( $this->hook_service->apply_filters( 'c3_disabled_cron_retry', false ) ) {
			if ( $this->log_cron_register_task ) {
				error_log( '===== C3 Invalidation cron has been SKIPPED [Disabled] ===' );
			}
			return false;
		}
		$invalidation_batch = $this->transient_service->load_invalidation_query();
		if ( $this->log_cron_register_task ) {
			error_log( print_r( $invalidation_batch, true ) );
		}
		if ( ! $invalidation_batch || empty( $invalidation_batch ) ) {
			if ( $this->log_cron_register_task ) {
				error_log( '===== C3 Invalidation cron has been SKIPPED [No Target Item] ===' );
			}
			return false;
		}
		$distribution_id = $this->cf_service->get_distribution_id();
		$query           = array(
			'DistributionId'    => esc_attr( $distribution_id ),
			'InvalidationBatch' => $invalidation_batch,
		);
		if ( $this->log_cron_register_task ) {
			error_log( print_r( $query, true ) );
		}

		/**
		 * Execute the invalidation.
		 */
		$result = $this->cf_service->create_invalidation( $query );
		if ( $this->log_cron_register_task ) {
			if ( is_wp_error( $result ) ) {
				error_log( 'C3 Cron: Invalidation failed: ' . $result->get_error_message() );
			} else {
				error_log( 'C3 Cron: Invalidation completed successfully' );
			}
		}
		$this->transient_service->delete_invalidation_query();
		if ( $this->log_cron_register_task ) {
			error_log( '===== C3 Invalidation cron has been COMPLETED ===' );
		}
		return true;
	}

	/**
	 * Get debug setting value
	 *
	 * @param string $setting_key Debug setting key.
	 * @return boolean Debug setting value.
	 */
	private function get_debug_setting( $setting_key ) {
		$debug_options = get_option( Constants::DEBUG_OPTION_NAME, array() );
		$value = isset( $debug_options[ $setting_key ] ) ? $debug_options[ $setting_key ] : false;
		return $value;
	}
}
