<?php

namespace BlockEditorColors;


class CustomColorsService {

	private $color_cpt_slug = 'block_editor_color';
	private $custom_colors = [];
	private $disabled_custom_colors = [];
	private static $_instance = null;

	public function __construct() {
		$this->set_color_cpt();
		$this->set_colors();

		add_action( 'admin_post_add_custom_color', array( $this, 'add_color' ) );
		add_action( 'admin_post_edit_custom_color', array( $this, 'edit_color' ) );
		add_action( 'admin_post_edit_inactive_color', array( $this, 'edit_inactive_color' ) );
		if ( wp_doing_ajax() ) {
			add_action( 'wp_ajax_bec_update_color_order', array( $this, 'update_color_order' ) );
		}
	}

	public static function getInstance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	public function set_color_cpt() {
		register_post_type( $this->color_cpt_slug, array(
			'label'  => esc_html__( 'Editor Color', 'block-editor-colors' ),
			'labels' => array(
				'name'          => esc_html__( 'Editor Colors', 'block-editor-colors' ),
				'singular_name' => esc_html__( 'Editor Color', 'block-editor-colors' ),
				'add_new'       => esc_html__( 'Add Editor Color', 'block-editor-colors' ),
				'add_new_item'  => esc_html__( 'Add Editor Color', 'block-editor-colors' ),
				'edit_item'     => esc_html__( 'Edit Editor Color', 'block-editor-colors' ),
				'new_item'      => esc_html__( 'New Editor Color', 'block-editor-colors' ),
				'view_item'     => esc_html__( 'View Editor Color', 'block-editor-colors' ),
				'search_items'  => esc_html__( 'Find Editor Color', 'block-editor-colors' ),
				'not_found'     => esc_html__( 'Editor Color Not Found', 'block-editor-colors' ),
				'menu_name'     => esc_html__( 'Editor Colors', 'block-editor-colors' ),
			),
			'public' => false,
		) );
	}

	private function set_colors() {

		$args = array(
			'post_type'   => $this->color_cpt_slug,
			'orderby'     => 'menu_order',
			'order'       => 'ASC',
			'post_status' => 'any',
			'numberposts' => -1
		);

		$posts           = get_posts( $args );
		$colors          = [];
		$disabled_colors = [];

		foreach ( $posts as $post ) {
			if ( $post->post_status === 'publish' ) {
				$colors[ $post->ID ] = [
					'name'  => $post->post_title,
					'slug'  => get_post_meta( $post->ID, 'slug', true ),
					'color' => get_post_meta( $post->ID, 'color', true ),
				];
			} else {
				$disabled_colors[ $post->ID ] = [
					'name'  => $post->post_title,
					'slug'  => get_post_meta( $post->ID, 'slug', true ),
					'color' => get_post_meta( $post->ID, 'color', true ),
				];
			}
		}

		$this->custom_colors          = $colors;
		$this->disabled_custom_colors = $disabled_colors;
	}

	public function get_colors( $disabled = false ) {
		if ( $disabled ) {
			return $this->disabled_custom_colors;
		}

		return $this->custom_colors;
	}

	public function add_color() {

		if ( ! isset( $_POST['create_custom_color_nonce'] ) || ! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['create_custom_color_nonce'] ) ), 'create_custom_color' ) ) {
			wp_die( esc_html__( 'Denied', 'block-editor-colors' ) );
		}

		if ( ! isset( $_POST['new_name'] ) || ! isset( $_POST['new_slug'] ) || ! isset( $_POST['new_color'] ) ) {
			wp_die( esc_html__( 'Empty fields', 'block-editor-colors' ) );
		}

		$name  = sanitize_text_field( wp_unslash( $_POST['new_name'] ) );
		$slug  = sanitize_title( wp_unslash( $_POST['new_slug'] ) );
		$color = sanitize_hex_color( wp_unslash( $_POST['new_color'] ) );

		$this->update_color( $name, $color, $slug );

		wp_redirect( SettingsPage::getAdminUrl() );
		exit;
	}

	public function edit_color() {

		if ( ! isset( $_POST['update_custom_color_nonce'] ) || ! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['update_custom_color_nonce'] ) ), 'update_custom_color' ) ) {
			wp_die( esc_html__( 'Denied', 'block-editor-colors' ) );
		}

		if ( ! isset( $_POST['color_id'] ) ) {
			wp_die( esc_html__( 'You must specify Color ID', 'block-editor-colors' ) );
		}

		$id = absint( $_POST['color_id'] );

		if ( isset( $_POST['disable'] ) ) {
			$this->disable_color( $id );
			wp_redirect( SettingsPage::getAdminUrl() );
			exit;
		}

		if ( ! isset( $_POST['name'] ) || ! isset( $_POST['color'] ) || ! isset( $_POST['update'] ) ) {
			wp_die( esc_html__( 'Empty fields', 'block-editor-colors' ) );
		}

		$name  = sanitize_text_field( wp_unslash( $_POST['name'] ) );
		$color = sanitize_hex_color( wp_unslash( $_POST['color'] ) );

		$this->update_color( $name, $color, false, $id );

		wp_redirect( SettingsPage::getAdminUrl() );
		exit;

	}

	private function disable_color( $id ) {
		wp_update_post( array(
			'ID'          => $id,
			'post_type'   => $this->color_cpt_slug,
			'post_status' => 'draft',
		) );
	}

	private function enable_color( $id ) {
		wp_update_post( array(
			'ID'          => $id,
			'post_type'   => $this->color_cpt_slug,
			'post_status' => 'publish',
		) );
	}

	private function delete_color( $id ) {
		wp_delete_post( $id );
	}

	private function update_color( $name, $color, $slug = false, $id = false ) {

		$post_data = array(
			'post_type'   => $this->color_cpt_slug,
			'post_title'  => $name,
			'meta_input'  => array(
				'color' => $color,
			),
			'post_status' => 'publish',
		);

		if ( $slug ) {
			$post_data['meta_input']['slug'] = $slug;
		}

		if ( $id ) {
			$post_data['ID'] = $id;
		}

		wp_insert_post( $post_data );
	}

	public function edit_inactive_color() {
		if ( ! isset( $_POST['update_inactive_color_nonce'] ) || ! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['update_inactive_color_nonce'] ) ), 'update_inactive_color' ) ) {
			wp_die( esc_html__( 'Denied', 'block-editor-colors' ) );
		}

		if ( ! isset( $_POST['color_id'] ) ) {
			wp_die( esc_html__( 'You must specify Color ID', 'block-editor-colors' ) );
		}

		$id = absint( $_POST['color_id'] );

		if ( isset( $_POST['delete'] ) ) {
			$this->delete_color( $id );
		}

		if ( isset( $_POST['restore'] ) ) {
			$this->enable_color( $id );
		}

		wp_redirect( SettingsPage::getAdminUrl() );
		exit;
	}

	public function update_color_order() {
		check_ajax_referer( 'block_editor_colors_nonce', 'nonce' );

		$colors = $this->recursive_sanitize_array( wp_unslash( $_POST['colors'] ) ); // phpcs:ignore

		foreach ( $colors as $order => $color_id ) {
			$updated = wp_update_post( [
				'ID'         => $color_id,
				'menu_order' => $order
			] );

			if ( ! $updated ) {
				wp_send_json_error();
			}
		}

		wp_send_json_success();
		wp_die();
	}

	/**
	 * Recursive sanitation for an array
	 *
	 * @since 1.2.1
	 *
	 * @param $array
	 *
	 * @return mixed
	 */
	function recursive_sanitize_array( $array ) {

		foreach ( $array as $key => &$value ) {
			if ( is_array( $value ) ) {
				$value = $this->recursive_sanitize_array( $value );
			}
			else {
				$value = sanitize_text_field( $value );
			}
		}

		return $array;
	}


}

CustomColorsService::getInstance();