<?php
/**
 * Display module
 *
 * @package Better_Search
 */

namespace WebberZone\Better_Search\Frontend;

use WebberZone\Better_Search\Util\Helpers;
use WebberZone\Better_Search\Util\Hook_Registry;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Display Class.
 *
 * @since 3.3.0
 */
class Display {

	/**
	 * Indicates if the current content is the primary content.
	 *
	 * @since 4.0.2
	 * @var int
	 */
	private static $title_count = 0;

	/**
	 * Constructor class.
	 *
	 * @since 3.3.0
	 */
	public function __construct() {
		Hook_Registry::add_filter( 'the_content', array( $this, 'content' ), 999 );
		Hook_Registry::add_filter( 'get_the_excerpt', array( $this, 'content' ), 999 );
		Hook_Registry::add_filter( 'the_title', array( $this, 'content' ), 999 );
		Hook_Registry::add_filter( 'the_bsearch_excerpt', array( $this, 'content' ), 999 );
	}

	/**
	 * Highlight search queries in the_content.
	 *
	 * @since 3.3.0
	 *
	 * @param string $content Post content.
	 *
	 * @return string Post Content
	 */
	public function content( $content ) {
		if ( is_admin() || ! in_the_loop() ) {
			return $content;
		}
		$referer = wp_get_referer() ? urldecode( wp_get_referer() ) : '';
		if ( is_search() ) {
			$is_referer_search_engine = true;
		} else {
			$siteurl = get_option( 'home' );
			if ( preg_match( "#^$siteurl#i", $referer ) ) {
				parse_str( (string) wp_parse_url( $referer, PHP_URL_QUERY ), $queries );
				if ( ! empty( $queries['s'] ) || preg_match( '#/search/.*#i', $referer ) ) {
					$is_referer_search_engine = true;
				}
			}
		}

		if ( empty( $is_referer_search_engine ) ) {
			return $content;
		}

		if ( bsearch_get_option( 'highlight' ) && is_search() ) {
			$search_query = get_bsearch_query();
		} elseif ( bsearch_get_option( 'highlight_followed_links' ) ) {
			if ( preg_match( '/\/search\/([^\/\?]+)/i', $referer, $matches ) ) {
				$search_query = $matches[1];
			} else {
				$search_query = preg_replace( '/^.*s=([^&]+)&?.*$/i', '$1', $referer );
			}
			$search_query = preg_replace( '/\'|"/', '', $search_query );

			if ( current_filter() === 'the_title' ) {
				if ( self::$title_count > 0 ) {
					return $content;
				}
				++self::$title_count;
			}
		}

		if ( ! empty( $search_query ) ) {
			$search_query = str_replace( array( "'", '"', '&quot;', '\+', '\-' ), '', $search_query );
			$keys         = preg_split( '/[\s,\+\.]+/', $search_query );
			$content      = Helpers::highlight( $content, $keys );
		}

		return $content;
	}
}
