<?php

namespace Baqend\WordPress;

use Baqend\WordPress\Controller\Controller;

/**
 * Register all actions and filters for the plugin.
 */
class Loader {

    /**
     * The array of actions registered with WordPress.
     *
     * @var array The actions registered with WordPress to fire when the plugin loads
     */
    protected $actions;

    /**
     * The array of filters registered with WordPress.
     *
     * @var array The filters registered with WordPress to fire when the plugin loads
     */
    protected $filters;

    public function __construct( Plugin $plugin ) {
        $this->actions = [];
        $this->filters = [];
    }

    /**
     * Adds a new controller which handles actions.
     *
     * @param Controller $controller
     */
    public function add_controller( Controller $controller ) {
        $controller->register( $this );
    }

    /**
     * Add a new action to the collection to be registered with WordPress.
     *
     * @param string $hook The name of the WordPress action that is being registered
     * @param callable $callback The name of the function definition on the $component
     * @param int $priority Optional. The priority at which the function should be fired. Default is 10
     * @param int $accepted_args Optional. The number of arguments that should be passed to the $callback. Default is 1
     */
    public function add_action( $hook, callable $callback, $priority = 10, $accepted_args = 1 ) {
        $this->actions = $this->add( $this->actions, $hook, $callback, $priority, $accepted_args );
    }

    /**
     * Add a new AJAX action to the collection to be registered with WordPress.
     *
     * @param string $hook The name of the WordPress AJAX hook that is being registered
     * @param callable $callback The name of the function definition on the $component
     * @param int $accepted_args Optional. The number of arguments that should be passed to the $callback. Default is 1
     * @param int $priority Optional. The priority at which the function should be fired. Default is 10
     */
    public function add_ajax( $hook, callable $callback, $accepted_args = 1, $priority = 10 ) {
        $this->add_action( 'wp_ajax_baqend_' . $hook, $callback, $priority, $accepted_args );
    }

    /**
     * Add a new filter to the collection to be registered with WordPress.
     *
     * @param string $hook The name of the WordPress filter that is being registered
     * @param callable $callback The name of the function definition on the $component
     * @param int $priority Optional. he priority at which the function should be fired. Default is 10
     * @param int $accepted_args Optional. The number of arguments that should be passed to the $callback. Default is 1
     */
    public function add_filter( $hook, callable $callback, $priority = 10, $accepted_args = 1 ) {
        $this->filters = $this->add( $this->filters, $hook, $callback, $priority, $accepted_args );
    }

    /**
     * A utility function that is used to register the actions and hooks into a single collection.
     *
     * @param array $hooks The collection of hooks that is being registered (that is, actions or filters)
     * @param string $hook The name of the WordPress filter that is being registered
     * @param callable $callback The name of the function definition on the $component
     * @param int $priority The priority at which the function should be fired
     * @param int $accepted_args The number of arguments that should be passed to the $callback
     *
     * @return array The collection of actions and filters registered with WordPress
     */
    private function add( $hooks, $hook, callable $callback, $priority, $accepted_args ) {
        $hooks[] = [
            'hook'          => $hook,
            'callback'      => $callback,
            'priority'      => $priority,
            'accepted_args' => $accepted_args,
        ];

        return $hooks;
    }

    /**
     * Register the filters and actions with WordPress.
     */
    public function run() {
        foreach ( $this->filters as $hook ) {
            add_filter( $hook['hook'], $hook['callback'], $hook['priority'], $hook['accepted_args'] );
        }
        foreach ( $this->actions as $hook ) {
            add_action( $hook['hook'], $hook['callback'], $hook['priority'], $hook['accepted_args'] );
        }
    }
}
