<?php
namespace Lex\Settings\V2;

/**
 * Lex Settings Framework
 *
 * A comprehensive WordPress settings system with tabbed interface,
 * field rendering, widgets, and responsive design.
 *
 * @version 2.0.0
 * @author LeanPlugins
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Lex Settings Main Class
 */
class Settings {

    /**
     * Registry of all Settings instances
     * 
     * @var array<string, Settings>
     */
    private static $instances = [];

    /**
     * Unique instance identifier
     * 
     * @var string
     */
    private $instance_id;

    /**
     * Configuration array
     * 
     * @var array
     */
    private $config = [];

    /**
     * Registered tabs array
     * 
     * @var array
     */
    private $registered_tabs = [];

    /**
     * Service instances (will be initialized in Phase 2)
     * 
     * @var object|null
     */
    public $dataManager = null;
    public $fieldRenderer = null;
    public $sectionRenderer = null;
    public $widgetRenderer = null;
    public $ajaxHandler = null;
    public $menu = null;
    public $assetManager = null;

    /**
     * Constructor
     *
     * @param array $config Configuration array
     */
    public function __construct($config = []) {
        // Parse and set configuration (defaults and user-provided)
        $this->config = $this->parseConfig($config);
        
        // Generate instance ID
        $this->instance_id = $this->config['instance_id'];
        
        // Store in registry
        self::$instances[$this->instance_id] = $this;
        
        // Initialize services
        $this->initServices();
        
        // Auto-activate demo if examples directory exists
        $this->maybeActivateDemo();
        
        // Register WordPress hooks
        $this->registerHooks();
    }
    
    /**
     * Initialize all service classes
     * 
     * @return void
     */
    private function initServices() {
        // Load utility classes first (used by services)
        require_once __DIR__ . '/includes/utilities/class-dot-notation.php';
        require_once __DIR__ . '/includes/utilities/class-sanitizer.php';
        require_once __DIR__ . '/includes/utilities/class-validator.php';
        require_once __DIR__ . '/includes/utilities/class-security.php';
        
        // Load service classes
        require_once __DIR__ . '/includes/class-data-manager.php';
        require_once __DIR__ . '/includes/class-field-renderer.php';
        require_once __DIR__ . '/includes/class-section-renderer.php';
        require_once __DIR__ . '/includes/class-widget-renderer.php';
        require_once __DIR__ . '/includes/class-ajax-handler.php';
        require_once __DIR__ . '/includes/class-menu.php';
        require_once __DIR__ . '/includes/class-assets-manager.php';
        
        // Initialize service instances
        $this->dataManager = new Services\DataManager($this);
        $this->fieldRenderer = new Services\FieldRenderer($this);
        $this->sectionRenderer = new Services\SectionRenderer($this);
        $this->widgetRenderer = new Services\WidgetRenderer($this);
        $this->ajaxHandler = new Services\AjaxHandler($this);
        $this->menu = new Services\Menu($this);
        $this->assetManager = new Services\AssetManager($this);
    }
    
    /**
     * Register WordPress hooks
     * 
     * @return void
     */
    private function registerHooks() {
        // Register settings with WordPress Settings API
        add_action('admin_init', [$this, 'registerSettings']);
        
        // Register admin menu if enabled
        $this->menu->register();
    }
    
    /**
     * Register settings with WordPress Settings API
     * 
     * @return void
     */
    public function registerSettings() {
        $option_key = $this->getConfig('option_key');
        $settings_group = $this->getConfig('settings_group');
        
        register_setting(
            $settings_group,
            $option_key,
            [
                'sanitize_callback' => [$this, 'sanitizeSettingsCallback'],
                'default' => []
            ]
        );
    }
    
    /**
     * Sanitize settings callback
     * 
     * @param array $input Raw input data
     * @return array Sanitized data
     */
    public function sanitizeSettingsCallback($input) {
        return Utilities\Sanitizer::sanitizeSettings($input);
    }

    /**
     * Parse configuration and generate instance-specific values
     *
     * Merges user-provided configuration with defaults and generates
     * instance-specific values (instance_id, ajax_prefix, js_global, etc.)
     * based on menu_slug or provided instance_id.
     *
     * @param array $config User-provided configuration array. Available parameters:
     *   - instance_id (string|null): Unique identifier for this Settings instance.
     *                                 If not provided, generated from menu_slug.
     *                                 Default: sanitize_key(menu_slug)
     *   - menu_slug (string): WordPress admin menu slug. Default: 'lex-settings'
     *   - option_key (string|null): WordPress option key for storing settings.
     *                                Default: '{instance_id}_options'
     *   - page_title (string): Settings page title. Default: 'Lex Settings'
     *   - menu_title (string): Admin menu item title. Default: 'Lex Settings'
     *   - capability (string): Required user capability. Default: 'manage_options'
     *   - icon (string): Dashicon class name. Default: 'dashicons-admin-settings'
     *   - position (int): Admin menu position. Default: 30
     *   - register_menu (bool): Whether to auto-register admin menu. Default: true
     *   - version (string): Framework version. Default: '2.0.0'
     *   - debug_mode (bool): Enable debug logging. Default: true
     *   - defaults (array|null): Default settings array. Default: null
     *   - logo (string): SVG logo string or dashicon. Default: gear icon SVG
     *   - ajax_prefix (string|null): Prefix for AJAX action names.
     *                                 Default: '{instance_id}_'
     *   - js_global (string|null): JavaScript global variable name (camelCase).
     *                              Default: '{instance_id}Settings' (camelCase)
     *   - settings_group (string|null): WordPress settings group name.
     *                                   Default: '{instance_id}_settings_group'
 *   - nonce_action (string|null): Nonce action name.
 *                                 Default: '{instance_id}_nonce'
 *   - allowed_pages (array): Additional pages where assets should load.
 *                             Supports multiple formats:
 *                             - Page slug: 'my-page' (from $_GET['page'])
 *                             - Hook suffix: 'toplevel_page_my-page'
 *                             - Wildcard pattern: 'my-plugin-*' (matches any page starting with pattern)
 *                             Default: []
 *   - page_check_callback (callable|null): Custom callback function for page checking.
 *                                          If provided, this takes precedence over allowed_pages.
 *                                          Callback receives $hook parameter and should return bool.
 *                                          Example: function($hook) { return leanpl_is_our_admin_page($hook); }
 *                                          Default: null
 *   - dropdown_label (string): Label for the dropdown menu when tabs have dropdown=true.
 *                              Default: 'More'
 *
 * @return array Parsed configuration array with all defaults applied and
     *               instance-specific values generated.
     */
    private function parseConfig($config) {
        // Default SVG logo (simple settings/gear icon)
        $default_logo_svg = '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M12 15.5C13.933 15.5 15.5 13.933 15.5 12C15.5 10.067 13.933 8.5 12 8.5C10.067 8.5 8.5 10.067 8.5 12C8.5 13.933 10.067 15.5 12 15.5Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M19.4 15C19.2669 15.3016 19.2272 15.6362 19.286 15.9606C19.3448 16.285 19.4995 16.5843 19.73 16.82L19.79 16.88C19.976 17.0657 20.1235 17.2863 20.2241 17.5291C20.3248 17.7719 20.3766 18.0322 20.3766 18.295C20.3766 18.5578 20.3248 18.8181 20.2241 19.0609C20.1235 19.3037 19.976 19.5243 19.79 19.71C19.6043 19.896 19.3837 20.0435 19.1409 20.1441C18.8981 20.2448 18.6378 20.2966 18.375 20.2966C18.1122 20.2966 17.8519 20.2448 17.6091 20.1441C17.3663 20.0435 17.1457 19.896 16.96 19.71L16.9 19.65C16.6643 19.4195 16.365 19.2648 16.0406 19.206C15.7162 19.1472 15.3816 19.1869 15.08 19.32C14.7842 19.4468 14.532 19.6572 14.3543 19.9255C14.1766 20.1938 14.0813 20.5082 14.08 20.83V21C14.08 21.5304 13.8693 22.0391 13.4942 22.4142C13.1191 22.7893 12.6104 23 12.08 23C11.5496 23 11.0409 22.7893 10.6658 22.4142C10.2907 22.0391 10.08 21.5304 10.08 21V20.91C10.0723 20.579 9.96512 20.258 9.77251 19.9887C9.5799 19.7194 9.31074 19.5143 9 19.4C8.69838 19.2669 8.36381 19.2272 8.03941 19.286C7.71502 19.3448 7.41568 19.4995 7.18 19.73L7.12 19.79C6.93425 19.976 6.71368 20.1235 6.47088 20.2241C6.22808 20.3248 5.96783 20.3766 5.705 20.3766C5.44217 20.3766 5.18192 20.3248 4.93912 20.2241C4.69632 20.1235 4.47575 19.976 4.29 19.79C4.10405 19.6043 3.95653 19.3837 3.85588 19.1409C3.75523 18.8981 3.70343 18.6378 3.70343 18.375C3.70343 18.1122 3.75523 17.8519 3.85588 17.6091C3.95653 17.3663 4.10405 17.1457 4.29 16.96L4.35 16.9C4.58054 16.6643 4.73519 16.365 4.794 16.0406C4.85282 15.7162 4.81312 15.3816 4.68 15.08C4.55324 14.7842 4.34276 14.532 4.07447 14.3543C3.80618 14.1766 3.49179 14.0813 3.17 14.08H3C2.46957 14.08 1.96086 13.8693 1.58579 13.4942C1.21071 13.1191 1 12.6104 1 12.08C1 11.5496 1.21071 11.0409 1.58579 10.6658C1.96086 10.2907 2.46957 10.08 3 10.08H3.09C3.42099 10.0723 3.742 9.96512 4.0113 9.77251C4.28059 9.5799 4.48572 9.31074 4.6 9C4.73312 8.69838 4.77282 8.36381 4.714 8.03941C4.65519 7.71502 4.50054 7.41568 4.27 7.18L4.21 7.12C4.02405 6.93425 3.87653 6.71368 3.77588 6.47088C3.67523 6.22808 3.62343 5.96783 3.62343 5.705C3.62343 5.44217 3.67523 5.18192 3.77588 4.93912C3.87653 4.69632 4.02405 4.47575 4.21 4.29C4.39575 4.10405 4.61632 3.95653 4.85912 3.85588C5.10192 3.75523 5.36217 3.70343 5.625 3.70343C5.88783 3.70343 6.14808 3.75523 6.39088 3.85588C6.63368 3.95653 6.85425 4.10405 7.04 4.29L7.1 4.35C7.33568 4.58054 7.63502 4.73519 7.95941 4.794C8.28381 4.85282 8.61838 4.81312 8.92 4.68H9C9.29577 4.55324 9.54802 4.34276 9.72569 4.07447C9.90337 3.80618 9.99872 3.49179 10 3.17V3C10 2.46957 10.2107 1.96086 10.5858 1.58579C10.9609 1.21071 11.4696 1 12 1C12.5304 1 13.0391 1.21071 13.4142 1.58579C13.7893 1.96086 14 2.46957 14 3V3.09C14.0013 3.41179 14.0966 3.72618 14.2743 3.99447C14.452 4.26276 14.7042 4.47324 15 4.6C15.3016 4.73312 15.6362 4.77282 15.9606 4.714C16.285 4.65519 16.5843 4.50054 16.82 4.27L16.88 4.21C17.0657 4.02405 17.2863 3.87653 17.5291 3.77588C17.7719 3.67523 18.0322 3.62343 18.295 3.62343C18.5578 3.62343 18.8181 3.67523 19.0609 3.77588C19.3037 3.87653 19.5243 4.02405 19.71 4.21C19.896 4.39575 20.0435 4.61632 20.1441 4.85912C20.2448 5.10192 20.2966 5.36217 20.2966 5.625C20.2966 5.88783 20.2448 6.14808 20.1441 6.39088C20.0435 6.63368 19.896 6.85425 19.71 7.04L19.65 7.1C19.4195 7.33568 19.2648 7.63502 19.206 7.95941C19.1472 8.28381 19.1869 8.61838 19.32 8.92V9C19.4468 9.29577 19.6572 9.54802 19.9255 9.72569C20.1938 9.90337 20.5082 9.99872 20.83 10H21C21.5304 10 22.0391 10.2107 22.4142 10.5858C22.7893 10.9609 23 11.4696 23 12C23 12.5304 22.7893 13.0391 22.4142 13.4142C22.0391 13.7893 21.5304 14 21 14H20.91C20.5882 14.0013 20.2738 14.0966 20.0055 14.2743C19.7372 14.452 19.5268 14.7042 19.4 15Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>';
        
        // Get menu_slug for instance_id generation if not provided
        $menu_slug = isset($config['menu_slug']) ? $config['menu_slug'] : 'lex-settings';
        
        // Generate instance_id from menu_slug if not provided
        $instance_id = isset($config['instance_id']) 
            ? sanitize_key($config['instance_id'])
            : sanitize_key($menu_slug);
        
        // Generate instance-specific values
        $ajax_prefix = isset($config['ajax_prefix']) 
            ? $config['ajax_prefix'] 
            : $instance_id . '_';
        
        // Generate JS global name (convert instance_id to camelCase)
        $js_global = isset($config['js_global']) 
            ? $config['js_global']
            : $this->generateJsGlobalName($instance_id);
        
        // Generate settings group
        $settings_group = isset($config['settings_group'])
            ? $config['settings_group']
            : $instance_id . '_settings_group';
        
        // Generate nonce action
        $nonce_action = isset($config['nonce_action'])
            ? $config['nonce_action']
            : $instance_id . '_nonce';
        
        // Default configuration
        $defaults = [
            'instance_id'      => $instance_id,
            'option_key'       => isset($config['option_key']) ? $config['option_key'] : $instance_id . '_options',
            'menu_slug'        => $menu_slug,
            'page_title'       => __('Lex Settings', 'lex-settings'),
            'menu_title'       => __('Lex Settings', 'lex-settings'),
            'capability'       => 'manage_options',
            'icon'             => 'dashicons-admin-settings',
            'position'         => 30,
            'register_menu'    => true,
            'version'          => '2.0.0',
            'debug_mode'       => true,
            'defaults'         => isset($config['defaults']) ? $config['defaults'] : null,
            'logo'             => $default_logo_svg,
            'ajax_prefix'      => $ajax_prefix,
            'js_global'        => $js_global,
            'settings_group'   => $settings_group,
            'nonce_action'     => $nonce_action,
            'allowed_pages'    => isset($config['allowed_pages']) ? $config['allowed_pages'] : [],
            'page_check_callback' => isset($config['page_check_callback']) && is_callable($config['page_check_callback']) ? $config['page_check_callback'] : null,
            'dropdown_label'   => isset($config['dropdown_label']) ? $config['dropdown_label'] : __('More', 'lex-settings'),
        ];
        
        // Merge user config with defaults
        $parsed = wp_parse_args($config, $defaults);
        
        // Set framework_path - use provided path or fallback to __DIR__
        // This allows each plugin instance to have its own framework path,
        // preventing conflicts when multiple plugins use the same framework version
        // framework_path should point to the core/ directory
        $parsed['framework_path'] = isset($config['framework_path']) 
            ? $config['framework_path'] 
            : __DIR__;
        
        // Set config_path - points to config/ directory (one level up from core/)
        $parsed['config_path'] = dirname($parsed['framework_path']) . '/config';
        
        return $parsed;
    }

    /**
     * Generate JavaScript global variable name from instance_id
     * 
     * Converts 'my_plugin' to 'myPluginSettings'
     * 
     * @param string $instance_id Instance identifier
     * @return string CamelCase JavaScript global name
     */
    private function generateJsGlobalName($instance_id) {
        // Convert snake_case or kebab-case to camelCase
        $parts = preg_split('/[-_]/', $instance_id);
        $camelCase = '';
        
        foreach ($parts as $i => $part) {
            if ($i === 0) {
                $camelCase .= strtolower($part);
            } else {
                $camelCase .= ucfirst(strtolower($part));
            }
        }
        
        return $camelCase . 'Settings';
    }

    /**
     * Auto-activate demo if examples directory exists
     */
    private function maybeActivateDemo() {
        $config_path = $this->getConfig('config_path');
        $examples_init = $config_path . '/examples/init.php';
        
        if (file_exists($examples_init)) {
            // Pass instance_id to demo init so it can register tabs
            // Using a constant that will be available in the demo init file
            if (!defined('LEX_SETTINGS_DEMO_INSTANCE_ID')) {
                define('LEX_SETTINGS_DEMO_INSTANCE_ID', $this->instance_id);
            }
            require_once $examples_init;
        }
    }

    /**
     * Get configuration value
     *
     * @param string|null $key Configuration key (null to get all config)
     * @return mixed Configuration value or array
     */
    public function getConfig($key = null) {
        if ($key === null) {
            return $this->config;
        }
        return isset($this->config[$key]) ? $this->config[$key] : null;
    }

    /**
     * Set configuration value
     *
     * @param string $key Configuration key
     * @param mixed $value Configuration value
     */
    public function setConfig($key, $value) {
        $this->config[$key] = $value;
    }

    /**
     * Get instance ID
     *
     * @return string Instance identifier
     */
    public function getInstanceId() {
        return $this->instance_id;
    }

    /**
     * Register a settings tab
     *
     * @param array $tab_config Tab configuration
     * @return Settings Instance for method chaining
     */
    public function registerTab($tab_config) {
        if (!isset($tab_config['id'])) {
            return $this;
        }
        
        $tab_id = $tab_config['id'];
        
        $this->registered_tabs[$tab_id] = wp_parse_args($tab_config, [
            'label' => ucfirst(str_replace(['-', '_'], ' ', $tab_id)),
            'icon' => '',
            'order' => 999,
            'pro' => false,
            'dropdown' => false,
        ]);
        
        // Sort tabs by order
        uasort($this->registered_tabs, function($a, $b) {
            return $a['order'] <=> $b['order'];
        });
        
        return $this;
    }

    /**
     * Get registered tabs
     *
     * @return array Registered tabs array
     */
    public function getTabs() {
        return $this->registered_tabs;
    }

    /**
     * Get Settings instance by ID
     *
     * @param string $instance_id Instance identifier
     * @return Settings|null Settings instance or null if not found
     */
    public static function getInstance($instance_id) {
        return isset(self::$instances[$instance_id]) ? self::$instances[$instance_id] : null;
    }

    /**
     * Get all registered instances
     *
     * @return array Array of Settings instances
     */
    public static function getAllInstances() {
        return self::$instances;
    }
}

