<?php
namespace LeanPL\Elementor;

use LeanPL\Assets_Manager;

/**
 * Elementor Integration Class
 * 
 * Handles ONLY Elementor-specific concerns:
 * - Compatibility checks
 * - Widget registration
 * - Admin notices
 * 
 * Asset loading delegated to Assets_Manager (DRY)
 */
class Integration {
    
    const MINIMUM_ELEMENTOR_VERSION = '2.0.0';
    const MINIMUM_PHP_VERSION = '5.6';

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Check if Elementor is available
        if (!did_action('elementor/loaded')) {
            return;
        }

        // Run compatibility checks
        if (!$this->is_compatible()) {
            return;
        }

        // Register ONLY widget-related hooks
        // Assets handled by Assets_Manager automatically
        $this->register_hooks();
    }

    private function is_compatible() {
        // Check Elementor version
        if (!defined('ELEMENTOR_VERSION') || 
            !version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_elementor_version']);
            return false;
        }

        // Check PHP version
        if (version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_php_version']);
            return false;
        }

        return true;
    }

    private function register_hooks() {
        // ONLY widget registration - Assets_Manager handles assets
        add_action('elementor/widgets/register', [$this, 'register_widgets']);
    }

    public function register_widgets($widgets_manager) {
        // Load widget files
        require_once LEANPL_DIR . '/includes/elementor/widgets/video-player.php';
        require_once LEANPL_DIR . '/includes/elementor/widgets/audio-player.php';

        // Register widgets
        $widgets_manager->register(new \LeanPL_Video_Player());
        $widgets_manager->register(new \LeanPL_Audio_Player());
    }

    public function admin_notice_minimum_php_version() {
        if (isset($_GET['activate'])) {
            unset($_GET['activate']);
        }

        $message = sprintf(
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'leanpl'),
            '<strong>' . esc_html__('Lean Player', 'leanpl') . '</strong>',
            '<strong>' . esc_html__('PHP', 'leanpl') . '</strong>',
            self::MINIMUM_PHP_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post($message));
    }

    public function admin_notice_minimum_elementor_version() {
        if (isset($_GET['activate'])) {
            unset($_GET['activate']);
        }

        $message = sprintf(
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'leanpl'),
            '<strong>' . esc_html__('Lean Player', 'leanpl') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'leanpl') . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post($message));
    }
}