<?php
namespace LeanPL;

/**
 * Unified Assets Manager
 * 
 * Single source of truth for ALL plugin assets across:
 * - Frontend (shortcodes, widgets)
 * - Elementor Editor
 * - Elementor Preview
 * - Admin pages
 * 
 * DRY Principle: All asset definitions, registration, and enqueuing in ONE place
 */
class Assets_Manager {

    private static $instance = null;
    private $version;
    private $assets_loaded = false;
    private $late_loading = false;

    /**
     * Asset definitions - SINGLE SOURCE OF TRUTH
     * All asset paths, handles, and dependencies defined here
     */
    private $asset_definitions = [
        'scripts' => [
            'plyr' => [
                'path' => '/assets/js/plyr.min.js',
                'deps' => ['jquery'],
                'footer' => true,
            ],
            'plyr-polyfilled' => [
                'path' => '/assets/js/plyr.polyfilled.min.js',
                'deps' => ['jquery'],
                'footer' => true,
            ],
            'leanpl-main' => [
                'path' => '/assets/js/main.js',
                'deps' => ['jquery', 'plyr'],
                'footer' => true,
            ],
            'leanpl-admin' => [
                'path' => '/assets/js/admin.js',
                'deps' => ['jquery'],
                'footer' => true,
            ],
        ],
        'styles' => [
            'plyr' => [
                'path' => '/assets/css/plyr.css',
                'deps' => [],
            ],
            'leanpl-main' => [
                'path' => '/assets/css/main.css',
                'deps' => ['plyr'],
            ],
            'leanpl-editor' => [
                'path' => '/assets/css/editor.css',
                'deps' => [],
            ],
            'leanpl-admin' => [
                'path' => '/assets/css/admin.css',
                'deps' => [],
            ],
        ],
    ];

    /**
     * Context definitions - what assets to load where
     */
    private $context_assets = [
        'frontend' => [
            'scripts' => ['plyr', 'leanpl-main'],
            'styles' => ['plyr', 'leanpl-main'],
        ],
        'elementor-editor' => [
            'scripts' => [],
            'styles' => ['leanpl-editor'],
        ],
        'elementor-preview' => [
            'scripts' => ['plyr', 'leanpl-main'],
            'styles' => ['plyr', 'leanpl-main'],
        ],
        'admin' => [
            'scripts' => ['leanpl-admin'],
            'styles' => ['leanpl-admin'],
        ],
    ];

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->init();
    }

    /**
     * Initialize - register ALL hooks in one place
     */
    public function init() {
        $this->version = leanpl_get_version();

        // Frontend hooks
        add_action('wp_enqueue_scripts', [$this, 'register_assets'], 10);
        add_action('wp_enqueue_scripts', [$this, 'conditional_enqueue'], 20);

        // Elementor hooks (if Elementor is active)
        if (did_action('elementor/loaded')) {
            add_action('elementor/frontend/after_register_scripts', [$this, 'register_assets']);
            add_action('elementor/frontend/after_enqueue_styles', [$this, 'enqueue_elementor_preview_assets']);
            add_action('elementor/editor/after_enqueue_styles', [$this, 'enqueue_elementor_editor_assets']);
        }

        // Admin hooks - Register assets first, then enqueue
        add_action('admin_enqueue_scripts', [$this, 'register_assets'] );
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }

    /**
     * Register ALL assets - DRY method using definitions array
     */
    public function register_assets() {
        // Register scripts
        foreach ($this->asset_definitions['scripts'] as $handle => $config) {
            if (!wp_script_is($handle, 'registered')) {
                wp_register_script(
                    $handle,
                    LEANPL_URI . $config['path'],
                    $config['deps'],
                    $this->version,
                    $config['footer']
                );
            }
        }

        // Register styles
        foreach ($this->asset_definitions['styles'] as $handle => $config) {
            if (!wp_style_is($handle, 'registered')) {
                wp_register_style(
                    $handle,
                    LEANPL_URI . $config['path'],
                    $config['deps'],
                    $this->version
                );
            }
        }
    }

    /**
     * Enqueue assets by context - DRY method
     */
    private function enqueue_by_context($context) {
        if (!isset($this->context_assets[$context])) {
            return;
        }

        $assets = $this->context_assets[$context];

        // Enqueue scripts
        foreach ($assets['scripts'] as $handle) {
            wp_enqueue_script($handle);
        }

        // Enqueue styles
        foreach ($assets['styles'] as $handle) {
            wp_enqueue_style($handle);
        }

        // Localize main script if it's being loaded
        if (in_array('leanpl-main', $assets['scripts'])) {
            $this->localize_main_script();
        }
    }

    /**
     * Early detection for shortcodes in post content
     */
    public function conditional_enqueue() {
        if ($this->assets_loaded || is_admin()) {
            return;
        }

        global $post;

        if ($post && 
            has_shortcode( $post->post_content, 'lean_video' ) ||
            has_shortcode( $post->post_content, 'lean_audio' ) || 
            has_shortcode( $post->post_content, 'lean_player' )
        ) {
            $this->enqueue_frontend_assets();
        }
    }

    /**
     * Enqueue frontend assets (shortcodes, widgets)
     */
    public function enqueue_frontend_assets() {
        if ($this->assets_loaded) {
            return;
        }

        $this->enqueue_by_context('frontend');
        $this->assets_loaded = true;
    }

    /**
     * Enqueue Elementor preview assets (iframe)
     */
    public function enqueue_elementor_preview_assets() {
        // Always load in Elementor preview - no conditional needed
        $this->enqueue_by_context('elementor-preview');
    }

    /**
     * Enqueue Elementor editor assets (editor UI)
     */
    public function enqueue_elementor_editor_assets() {
        // Always load in Elementor editor - no conditional needed
        $this->enqueue_by_context('elementor-editor');
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only on plugin's own admin pages
        if (!leanpl_is_our_admin_page()) {
            return;
        }

        $this->enqueue_by_context('admin');
    }

    /**
     * Localize main script - DRY method
     */
    private function localize_main_script() {
        static $localized = false;

        if ($localized) {
            return; // Only localize once
        }

        wp_localize_script('leanpl-main', 'lplData', [
            'version' => leanpl_get_version(),
            'debugMode' => leanpl_is_debug_mode() || leanpl_is_test_mode(),
            'baseVersion' => LEANPL_VERSION,
            'ajaxUrl' => admin_url('admin-ajax.php'),
        ]);

        $localized = true;
    }

    /**
     * Runtime safety net for late shortcode execution
     * Called from shortcode handler
     */
    public function ensure_assets_loaded() {
        if ($this->assets_loaded) {
            return;
        }

        // Check if we're in Elementor editor/preview
        if ($this->is_elementor_context()) {
            // Elementor handles asset loading via its own hooks
            return;
        }

        // Check timing
        if (did_action('wp_enqueue_scripts')) {
            // Too late - use fallback
            $this->late_loading = true;
            $this->inline_critical_assets();
        } else {
            // Still in time
            $this->enqueue_frontend_assets();
        }
    }

    /**
     * Detect if we're in Elementor context
     */
    private function is_elementor_context() {
        if (!did_action('elementor/loaded')) {
            return false;
        }

        // Check if in editor or preview mode
        if (\Elementor\Plugin::$instance->editor->is_edit_mode()) {
            return true;
        }

        if (\Elementor\Plugin::$instance->preview->is_preview_mode()) {
            return true;
        }

        return false;
    }

    /**
     * Fallback for late loading
     */
    private function inline_critical_assets() {
        if ($this->assets_loaded) {
            return;
        }

        add_action('wp_footer', [$this, 'output_critical_css'], 1);

        // Scripts can still be enqueued in footer
        foreach ($this->context_assets['frontend']['scripts'] as $handle) {
            wp_enqueue_script($handle);
        }

        $this->localize_main_script();
        $this->assets_loaded = true;
    }

    /**
     * Output critical CSS for late loading
     */
    public function output_critical_css() {
        if (!$this->late_loading) {
            return;
        }

        // Critical CSS inline
        echo '<style id="leanpl-critical-css">
            .lpl-player { opacity: 0; transition: opacity 0.3s ease; }
            .plyr.plyr--ready .lpl-player,
            .plyr--video .lpl-player,
            .plyr--audio .lpl-player { opacity: 1; }
            .lpl-player video, .lpl-player iframe { width: 100%; height: auto; }
        </style>';

        // Full CSS via link tags
        foreach ($this->context_assets['frontend']['styles'] as $handle) {
            $config = $this->asset_definitions['styles'][$handle];
            echo sprintf(
                '<link rel="stylesheet" id="%s-css" href="%s?ver=%s" type="text/css" media="all" />',
                esc_attr($handle),
                esc_url(LEANPL_URI . $config['path']),
                esc_attr($this->version)
            );
        }
    }

    /**
     * Public API: Check if specific asset is loaded
     */
    public function is_asset_loaded($handle) {
        return wp_script_is($handle, 'enqueued') || wp_style_is($handle, 'enqueued');
    }

    /**
     * Public API: Get asset URL
     */
    public function get_asset_url($handle, $type = 'script') {
        $key = ($type === 'script') ? 'scripts' : 'styles';
        
        if (!isset($this->asset_definitions[$key][$handle])) {
            return '';
        }

        return LEANPL_URI . $this->asset_definitions[$key][$handle]['path'];
    }
}

// Initialize
Assets_Manager::get_instance();