<?php

class ADFOIN_ZohoPeople extends Advanced_Form_Integration_OAuth2 {

    const authorization_endpoint = 'https://accounts.zoho.com/oauth/v2/auth';
    const token_endpoint         = 'https://accounts.zoho.com/oauth/v2/token';
    const refresh_token_endpoint = 'https://accounts.zoho.com/oauth/v2/token';

    public $data_center;
    public $state;
    private static $instance;

    public static function get_instance() {
        if ( empty( self::$instance ) ) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    private function __construct() {
        $this->authorization_endpoint = self::authorization_endpoint;
        $this->token_endpoint         = self::token_endpoint;
        $this->refresh_token_endpoint = self::refresh_token_endpoint;

        $option = (array) maybe_unserialize( get_option( 'adfoin_zohopeople_keys' ) );

        if ( isset( $option['data_center'] ) ) {
            $this->data_center = $option['data_center'];
        }

        if ( isset( $option['client_id'] ) ) {
            $this->client_id = $option['client_id'];
        }

        if ( isset( $option['client_secret'] ) ) {
            $this->client_secret = $option['client_secret'];
        }

        if ( isset( $option['access_token'] ) ) {
            $this->access_token = $option['access_token'];
        }

        if ( isset( $option['refresh_token'] ) ) {
            $this->refresh_token = $option['refresh_token'];
        }

        add_action( 'admin_init', array( $this, 'auth_redirect' ) );
        add_action( 'rest_api_init', array( $this, 'register_webhook_route' ) );

        add_filter( 'adfoin_action_providers', array( $this, 'register_actions' ), 10, 1 );
        add_filter( 'adfoin_settings_tabs', array( $this, 'register_settings_tab' ), 10, 1 );
        add_action( 'adfoin_settings_view', array( $this, 'settings_view' ), 10, 1 );
        add_action( 'adfoin_action_fields', array( $this, 'action_fields' ), 10, 1 );

        add_action( 'wp_ajax_adfoin_get_zohopeople_credentials', array( $this, 'ajax_get_credentials' ), 10, 0 );
        add_action( 'wp_ajax_adfoin_save_zohopeople_credentials', array( $this, 'ajax_save_credentials' ), 10, 0 );
        add_action( 'wp_ajax_adfoin_get_zohopeople_forms', array( $this, 'ajax_get_forms' ), 10, 0 );
        add_action( 'wp_ajax_adfoin_get_zohopeople_fields', array( $this, 'ajax_get_fields' ), 10, 0 );
    }

    public function register_webhook_route() {
        register_rest_route(
            'advancedformintegration',
            '/zohopeople',
            array(
                'methods'             => 'GET',
                'callback'            => array( $this, 'handle_oauth_callback' ),
                'permission_callback' => '__return_true',
            )
        );
    }

    public function handle_oauth_callback( $request ) {
        $params = $request->get_params();

        $code  = isset( $params['code'] ) ? sanitize_text_field( $params['code'] ) : '';
        $state = isset( $params['state'] ) ? sanitize_text_field( $params['state'] ) : '';

        if ( $code && $state ) {
            $redirect_to = add_query_arg(
                array(
                    'service' => 'authorize',
                    'action'  => 'adfoin_zohopeople_auth_redirect',
                    'code'    => $code,
                    'state'   => $state,
                ),
                admin_url( 'admin.php?page=advanced-form-integration-settings&tab=zohopeople' )
            );

            wp_safe_redirect( $redirect_to );
            exit();
        }

        return array( 'status' => 'ok' );
    }

    public function register_actions( $actions ) {
        $actions['zohopeople'] = array(
            'title' => __( 'Zoho People', 'advanced-form-integration' ),
            'tasks' => array(
                'create_employee' => __( 'Create Employee', 'advanced-form-integration' ),
            ),
        );

        return $actions;
    }

    public function register_settings_tab( $providers ) {
        $providers['zohopeople'] = __( 'Zoho People', 'advanced-form-integration' );

        return $providers;
    }

    public function settings_view( $current_tab ) {
        if ( 'zohopeople' !== $current_tab ) {
            return;
        }

        $title     = __( 'Zoho People', 'advanced-form-integration' );
        $key       = 'zohopeople';
        $arguments = wp_json_encode(
            array(
                'platform' => $key,
                'fields'   => array(
                    array(
                        'key'   => 'title',
                        'label' => __( 'Title', 'advanced-form-integration' ),
                        'hidden'=> false,
                    ),
                    array(
                        'key'   => 'dataCenter',
                        'label' => __( 'Data Center (com, eu, in, com.au, com.cn)', 'advanced-form-integration' ),
                        'hidden'=> false,
                    ),
                    array(
                        'key'   => 'clientId',
                        'label' => __( 'Client ID', 'advanced-form-integration' ),
                        'hidden'=> true,
                    ),
                    array(
                        'key'   => 'clientSecret',
                        'label' => __( 'Client Secret', 'advanced-form-integration' ),
                        'hidden'=> true,
                    ),
                    array(
                        'key'   => 'accessToken',
                        'label' => __( 'Access Token', 'advanced-form-integration' ),
                        'hidden'=> true,
                    ),
                    array(
                        'key'   => 'refreshToken',
                        'label' => __( 'Refresh Token', 'advanced-form-integration' ),
                        'hidden'=> true,
                    ),
                ),
            )
        );

        $instructions = sprintf(
            '<p>%s</p><ol><li>%s</li><li>%s</li><li>%s</li><li>%s</li><li>%s</li><li>%s</li></ol>',
            esc_html__( 'Follow these steps to connect your Zoho People account:', 'advanced-form-integration' ),
            esc_html__( 'Create a client in Zoho API Console with the Self Client type.', 'advanced-form-integration' ),
            esc_html__( 'Add the authorized redirect URI shown below to your client.', 'advanced-form-integration' ),
            esc_html__( 'Copy the Client ID and Client Secret into the form fields.', 'advanced-form-integration' ),
            esc_html__( 'Click Authorize to grant Advanced Form Integration access.', 'advanced-form-integration' ),
            esc_html__( 'Complete the OAuth authorization flow in the popup window.', 'advanced-form-integration' ),
            esc_html__( 'Save the settings once the account is marked as Connected.', 'advanced-form-integration' )
        );

        ?>
        <div class="wrap">
            <h1><?php echo esc_html( $title ); ?></h1>
            <p><strong><?php esc_html_e( 'Authorized Redirect URI:', 'advanced-form-integration' ); ?></strong> <code><?php echo esc_html( $this->get_redirect_uri() ); ?></code></p>
            <?php echo wp_kses_post( $instructions ); ?>
            <div id="api-key-management" data-arguments="<?php echo esc_attr( $arguments ); ?>"></div>
        </div>
        <?php
    }

    public function action_fields() {
        ?>
        <script type="text/template" id="zohopeople-action-template">
            <table class="form-table">
                <tr v-if="action.task == 'create_employee'">
                    <th scope="row"><?php esc_attr_e( 'Zoho People Account', 'advanced-form-integration' ); ?></th>
                    <td>
                        <select name="fieldData[credId]" v-model="fielddata.credId">
                            <option value=""><?php esc_html_e( 'Select Account...', 'advanced-form-integration' ); ?></option>
                            <?php $this->get_credentials_list(); ?>
                        </select>
                    </td>
                </tr>
                <tr v-if="action.task == 'create_employee'">
                    <th scope="row"><?php esc_attr_e( 'Target Form', 'advanced-form-integration' ); ?></th>
                    <td>
                        <input type="text" name="fieldData[formLinkName]" v-model="fielddata.formLinkName" placeholder="P_EmployeeView" />
                        <p class="description"><?php esc_html_e( 'Records are created in the Employee form (default: P_EmployeeView).', 'advanced-form-integration' ); ?></p>
                    </td>
                </tr>
                <editable-field
                    v-for="field in fields"
                    :key="field.value"
                    :field="field"
                    :trigger="trigger"
                    :action="action"
                    :fielddata="fielddata">
                </editable-field>
            </table>
        </script>
        <?php
    }

    public function ajax_get_credentials() {
        if ( ! adfoin_verify_nonce() ) {
            return;
        }

        $credentials = adfoin_read_credentials( 'zohopeople' );
        wp_send_json_success( $credentials );
    }

    public function ajax_save_credentials() {
        if ( ! adfoin_verify_nonce() ) {
            return;
        }

        $platform = sanitize_text_field( $_POST['platform'] ?? '' );

        if ( 'zohopeople' === $platform ) {
            $data = isset( $_POST['data'] ) ? adfoin_sanitize_text_or_array_field( $_POST['data'] ) : array();
            adfoin_save_credentials( $platform, $data );
        }

        wp_send_json_success();
    }

    public function ajax_get_forms() {
        if ( ! adfoin_verify_nonce() ) {
            return;
        }

        $cred_id = isset( $_POST['credId'] ) ? sanitize_text_field( $_POST['credId'] ) : '';

        $forms = $this->get_forms_list( $cred_id );

        if ( is_wp_error( $forms ) ) {
            wp_send_json_error( $forms->get_error_message() );
        }

        wp_send_json_success( $forms );
    }

    public function ajax_get_fields() {
        if ( ! adfoin_verify_nonce() ) {
            return;
        }

        $cred_id      = isset( $_POST['credId'] ) ? sanitize_text_field( $_POST['credId'] ) : '';
        $form_link    = isset( $_POST['formLinkName'] ) ? sanitize_text_field( $_POST['formLinkName'] ) : '';

        if ( ! $cred_id || ! $form_link ) {
            wp_send_json_success( array() );
        }

        $fields = $this->get_fields_list( $cred_id, $form_link );

        if ( is_wp_error( $fields ) ) {
            wp_send_json_error( $fields->get_error_message() );
        }

        wp_send_json_success( $fields );
    }

    public function get_forms_list( $cred_id ) {
        if ( ! $cred_id ) {
            return array();
        }

        $this->set_credentials( $cred_id );

        $response = $this->zohopeople_request( 'forms', 'GET' );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $body  = json_decode( wp_remote_retrieve_body( $response ), true );
        $forms = array();

        if ( isset( $body['forms'] ) && is_array( $body['forms'] ) ) {
            foreach ( $body['forms'] as $form ) {
                if ( isset( $form['formLinkName'], $form['displayName'] ) ) {
                    $forms[ $form['formLinkName'] ] = $form['displayName'];
                } elseif ( isset( $form['link_name'], $form['formname'] ) ) {
                    $forms[ $form['link_name'] ] = $form['formname'];
                }
            }
        } elseif ( isset( $body['data'] ) && is_array( $body['data'] ) ) {
            foreach ( $body['data'] as $form ) {
                if ( isset( $form['formLinkName'], $form['displayName'] ) ) {
                    $forms[ $form['formLinkName'] ] = $form['displayName'];
                }
            }
        }

        return $forms;
    }

    public function get_fields_list( $cred_id, $form_link ) {
        if ( ! $cred_id || ! $form_link ) {
            return array();
        }

        $this->set_credentials( $cred_id );

        $response = $this->zohopeople_request( 'forms/' . rawurlencode( $form_link ) . '/fields', 'GET' );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $body   = json_decode( wp_remote_retrieve_body( $response ), true );
        $fields = array();
        $raw    = array();

        if ( isset( $body['form_fields'] ) && is_array( $body['form_fields'] ) ) {
            $raw = $body['form_fields'];
        } elseif ( isset( $body['fields'] ) && is_array( $body['fields'] ) ) {
            $raw = $body['fields'];
        } elseif ( isset( $body['data'] ) && is_array( $body['data'] ) ) {
            $raw = $body['data'];
        }

        foreach ( $raw as $field ) {
            $link_name = $field['linkName'] ?? $field['link_name'] ?? '';
            $label     = $field['displayName'] ?? $field['labelName'] ?? $field['fieldLabel'] ?? '';

            if ( ! $link_name || ! $label ) {
                continue;
            }

            $fields[] = array(
                'key'         => $link_name,
                'value'       => $label,
                'description' => $field['helpText'] ?? '',
                'required'    => ! empty( $field['required'] ) || ( isset( $field['isMandatory'] ) && true === $field['isMandatory'] ),
                'controlType' => ! empty( $field['type'] ) && 'textarea' === strtolower( (string) $field['type'] ) ? 'textarea' : 'text',
            );
        }

        return $fields;
    }

    public function auth_redirect() {
        if ( empty( $_GET['service'] ) || 'authorize' !== $_GET['service'] ) {
            return;
        }

        $action = sanitize_text_field( $_GET['action'] ?? '' );

        if ( 'adfoin_zohopeople_auth_redirect' !== $action ) {
            return;
        }

        $state = sanitize_text_field( $_GET['state'] ?? '' );
        $code  = sanitize_text_field( $_GET['code'] ?? '' );

        if ( ! $state || ! $code ) {
            wp_safe_redirect( admin_url( 'admin.php?page=advanced-form-integration-settings&tab=zohopeople' ) );
            exit();
        }

        $credentials = adfoin_read_credentials( 'zohopeople' );

        foreach ( $credentials as $entry ) {
            if ( ( $entry['id'] ?? '' ) === $state ) {
                $this->client_id     = $entry['clientId'] ?? '';
                $this->client_secret = $entry['clientSecret'] ?? '';
                $this->data_center   = $entry['dataCenter'] ?? 'com';
                $this->state         = $state;
            }
        }

        $this->request_token( $code );

        wp_safe_redirect( admin_url( 'admin.php?page=advanced-form-integration-settings&tab=zohopeople' ) );
        exit();
    }

    protected function request_token( $authorization_code ) {
        $token_endpoint = $this->token_endpoint;

        if ( $this->data_center && 'com' !== $this->data_center ) {
            if ( 'com.au' === $this->data_center ) {
                $token_endpoint = 'https://accounts.zoho.com.au/oauth/v2/token';
            } elseif ( 'com.cn' === $this->data_center ) {
                $token_endpoint = 'https://accounts.zoho.com.cn/oauth/v2/token';
            } else {
                $token_endpoint = str_replace( 'com', $this->data_center, $token_endpoint );
            }
        }

        $endpoint = add_query_arg(
            array(
                'code'         => $authorization_code,
                'redirect_uri' => urlencode( $this->get_redirect_uri() ),
                'grant_type'   => 'authorization_code',
            ),
            $token_endpoint
        );

        $response = wp_remote_post(
            esc_url_raw( $endpoint ),
            array(
                'headers' => array(
                    'Authorization' => $this->get_http_authorization_header( 'basic' ),
                ),
            )
        );

        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['access_token'] ) ) {
            $this->access_token = $body['access_token'];
        }

        if ( isset( $body['refresh_token'] ) ) {
            $this->refresh_token = $body['refresh_token'];
        }

        $this->save_data();
    }

    protected function refresh_token() {
        if ( empty( $this->refresh_token ) ) {
            return;
        }

        $refresh_endpoint = $this->refresh_token_endpoint;

        if ( $this->data_center && 'com' !== $this->data_center ) {
            if ( 'com.au' === $this->data_center ) {
                $refresh_endpoint = 'https://accounts.zoho.com.au/oauth/v2/token';
            } elseif ( 'com.cn' === $this->data_center ) {
                $refresh_endpoint = 'https://accounts.zoho.com.cn/oauth/v2/token';
            } else {
                $refresh_endpoint = str_replace( 'com', $this->data_center, $refresh_endpoint );
            }
        }

        $endpoint = add_query_arg(
            array(
                'refresh_token' => $this->refresh_token,
                'grant_type'    => 'refresh_token',
            ),
            $refresh_endpoint
        );

        $response = wp_remote_post(
            esc_url_raw( $endpoint ),
            array(
                'headers' => array(
                    'Authorization' => $this->get_http_authorization_header( 'basic' ),
                ),
            )
        );

        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['access_token'] ) ) {
            $this->access_token = $body['access_token'];
        }

        if ( isset( $body['refresh_token'] ) ) {
            $this->refresh_token = $body['refresh_token'];
        }

        $this->save_data();
    }

    protected function save_data() {
        $credentials = adfoin_read_credentials( 'zohopeople' );

        foreach ( $credentials as &$entry ) {
            if ( ( $entry['id'] ?? '' ) === $this->state ) {
                $entry['accessToken']  = $this->access_token;
                $entry['refreshToken'] = $this->refresh_token;
            }
        }

        adfoin_save_credentials( 'zohopeople', $credentials );

        update_option(
            'adfoin_zohopeople_keys',
            maybe_serialize(
                array(
                    'data_center'   => $this->data_center,
                    'client_id'     => $this->client_id,
                    'client_secret' => $this->client_secret,
                    'access_token'  => $this->access_token,
                    'refresh_token' => $this->refresh_token,
                )
            )
        );
    }

    protected function get_redirect_uri() {
        return site_url( '/wp-json/advancedformintegration/zohopeople' );
    }

    public function set_credentials( $cred_id ) {
        $credentials = $this->get_credentials_by_id( $cred_id );

        if ( empty( $credentials ) ) {
            return;
        }

        $this->data_center   = $credentials['dataCenter'] ?? 'com';
        $this->client_id     = $credentials['clientId'] ?? '';
        $this->client_secret = $credentials['clientSecret'] ?? '';
        $this->access_token  = $credentials['accessToken'] ?? '';
        $this->refresh_token = $credentials['refreshToken'] ?? '';
        $this->state         = $credentials['id'] ?? '';
    }

    public function get_credentials_by_id( $cred_id ) {
        $all_credentials = adfoin_read_credentials( 'zohopeople' );

        if ( empty( $all_credentials ) || ! is_array( $all_credentials ) ) {
            return array();
        }

        foreach ( $all_credentials as $single ) {
            if ( $cred_id && ( $single['id'] ?? '' ) === $cred_id ) {
                return $single;
            }
        }

        return $all_credentials[0];
    }

    public function get_credentials_list() {
        $credentials = adfoin_read_credentials( 'zohopeople' );

        foreach ( $credentials as $option ) {
            printf(
                '<option value="%1$s">%2$s</option>',
                esc_attr( $option['id'] ),
                esc_html( $option['title'] )
            );
        }
    }

    public function zohopeople_request( $endpoint, $method = 'GET', $data = array(), $record = array(), $query = array() ) {
        static $refreshed = false;

        $base_url = 'https://people.zoho.com/people/api/';

        if ( $this->data_center && 'com' !== $this->data_center ) {
            if ( 'com.au' === $this->data_center ) {
                $base_url = 'https://people.zoho.com.au/people/api/';
            } elseif ( 'com.cn' === $this->data_center ) {
                $base_url = 'https://people.zoho.com.cn/people/api/';
            } else {
                $base_url = str_replace( 'com', $this->data_center, $base_url );
            }
        }

        $url = $base_url . ltrim( $endpoint, '/' );

        if ( ! empty( $query ) ) {
            $url = add_query_arg( $query, $url );
        }

        $args = array(
            'method'  => $method,
            'headers' => array(
                'Authorization' => 'Zoho-oauthtoken ' . $this->access_token,
                'Content-Type'  => 'application/json',
                'Accept'        => 'application/json',
            ),
        );

        if ( in_array( strtoupper( $method ), array( 'POST', 'PUT', 'PATCH' ), true ) && ! empty( $data ) ) {
            $args['body'] = wp_json_encode( $data );
        } elseif ( 'GET' === strtoupper( $method ) && ! empty( $data ) ) {
            $url = add_query_arg( $data, $url );
        }

        $response = wp_remote_request( esc_url_raw( $url ), $args );

        if ( 401 === wp_remote_retrieve_response_code( $response ) && ! $refreshed ) {
            $this->refresh_token();
            $refreshed = true;

            $args['headers']['Authorization'] = 'Zoho-oauthtoken ' . $this->access_token;
            $response = wp_remote_request( esc_url_raw( $url ), $args );
        }

        if ( $record ) {
            adfoin_add_to_log( $response, $url, $args, $record );
        }

        return $response;
    }
}

$adfoin_zohopeople = ADFOIN_ZohoPeople::get_instance();

function adfoin_zohopeople_fields() {
    return array(
        array( 'key' => 'employee_id', 'value' => __( 'Employee ID', 'advanced-form-integration' ), 'description' => '', 'required' => false, 'controlType' => 'text' ),
        array( 'key' => 'first_name', 'value' => __( 'First Name', 'advanced-form-integration' ), 'description' => '', 'required' => true, 'controlType' => 'text' ),
        array( 'key' => 'last_name', 'value' => __( 'Last Name', 'advanced-form-integration' ), 'description' => '', 'required' => true, 'controlType' => 'text' ),
        array( 'key' => 'email', 'value' => __( 'Email', 'advanced-form-integration' ), 'description' => '', 'required' => false, 'controlType' => 'text' ),
        array( 'key' => 'mobile', 'value' => __( 'Mobile', 'advanced-form-integration' ), 'description' => '', 'required' => false, 'controlType' => 'text' ),
    );
}

function adfoin_zohopeople_basic_field_map() {
    return array(
        'employee_id' => 'Employee_ID',
        'first_name'  => 'First_Name',
        'last_name'   => 'Last_Name',
        'email'       => 'Email',
        'mobile'      => 'Mobile',
    );
}

function adfoin_zohopeople_prepare_payload( $field_data, $posted_data, $field_map ) {
    $payload = array();

    foreach ( $field_map as $form_key => $api_key ) {
        if ( empty( $field_data[ $form_key ] ) ) {
            continue;
        }

        $value = adfoin_get_parsed_values( $field_data[ $form_key ], $posted_data );

        if ( '' === $value || null === $value ) {
            continue;
        }

        $payload[ $api_key ] = $value;
    }

    return $payload;
}

function adfoin_zohopeople_job_queue( $data ) {
    if ( ( $data['action_provider'] ?? '' ) !== 'zohopeople' || ( $data['task'] ?? '' ) !== 'create_employee' ) {
        return;
    }

    adfoin_zohopeople_send_data( $data['record'], $data['posted_data'] );
}

add_action( 'adfoin_job_queue', 'adfoin_zohopeople_job_queue', 10, 1 );

function adfoin_zohopeople_send_data( $record, $posted_data ) {
    $record_data = json_decode( $record['data'], true );

    if ( isset( $record_data['action_data']['cl'] ) && adfoin_check_conditional_logic( $record_data['action_data']['cl'], $posted_data ) ) {
        return;
    }

    $field_data = $record_data['field_data'] ?? array();
    $cred_id    = $field_data['credId'] ?? '';
    $form_link  = $field_data['formLinkName'] ?? 'P_EmployeeView';

    if ( ! $cred_id ) {
        return;
    }

    $payload = adfoin_zohopeople_prepare_payload( $field_data, $posted_data, adfoin_zohopeople_basic_field_map() );

    if ( empty( $payload ) ) {
        return;
    }

    $people = ADFOIN_ZohoPeople::get_instance();
    $people->set_credentials( $cred_id );

    $people->zohopeople_request(
        'forms/' . rawurlencode( $form_link ) . '/records',
        'POST',
        array( 'data' => array( $payload ) ),
        $record
    );
}
