<?php

add_filter( 'adfoin_action_providers', 'adfoin_sendlane_actions', 10, 1 );

function adfoin_sendlane_actions( $actions ) {

    $actions['sendlane'] = array(
        'title' => __( 'Sendlane', 'advanced-form-integration' ),
        'tasks' => array(
            'subscribe'   => __( 'Add/Update Contact', 'advanced-form-integration' ),
            'unsubscribe' => __( 'Unsubscribe Contact', 'advanced-form-integration' ),
        ),
    );

    return $actions;
}

add_filter( 'adfoin_settings_tabs', 'adfoin_sendlane_settings_tab', 10, 1 );

function adfoin_sendlane_settings_tab( $providers ) {
    $providers['sendlane'] = __( 'Sendlane', 'advanced-form-integration' );

    return $providers;
}

add_action( 'adfoin_settings_view', 'adfoin_sendlane_settings_view', 10, 1 );

function adfoin_sendlane_settings_view( $current_tab ) {
    if ( 'sendlane' !== $current_tab ) {
        return;
    }

    $nonce      = wp_create_nonce( 'adfoin_sendlane_settings' );
    $api_key    = get_option( 'adfoin_sendlane_api_key', '' );
    $api_secret = get_option( 'adfoin_sendlane_api_secret', '' );
    $subdomain  = get_option( 'adfoin_sendlane_subdomain', '' );
    ?>

    <form name="sendlane_save_form" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>"
          method="post" class="container">

        <input type="hidden" name="action" value="adfoin_save_sendlane_credentials">
        <input type="hidden" name="_nonce" value="<?php echo esc_attr( $nonce ); ?>"/>

        <table class="form-table">
            <tr valign="top">
                <th scope="row"> <?php esc_html_e( 'Sendlane API Key', 'advanced-form-integration' ); ?></th>
                <td>
                    <input type="text" name="adfoin_sendlane_api_key"
                           value="<?php echo esc_attr( $api_key ); ?>" placeholder="<?php esc_attr_e( 'Enter API Key', 'advanced-form-integration' ); ?>"
                           class="regular-text"/>
                </td>
            </tr>
            <tr valign="top">
                <th scope="row"> <?php esc_html_e( 'Sendlane API Secret', 'advanced-form-integration' ); ?></th>
                <td>
                    <input type="text" name="adfoin_sendlane_api_secret"
                           value="<?php echo esc_attr( $api_secret ); ?>" placeholder="<?php esc_attr_e( 'Enter API Secret', 'advanced-form-integration' ); ?>"
                           class="regular-text"/>
                </td>
            </tr>
            <tr valign="top">
                <th scope="row"> <?php esc_html_e( 'Sendlane Subdomain', 'advanced-form-integration' ); ?></th>
                <td>
                    <input type="text" name="adfoin_sendlane_subdomain"
                           value="<?php echo esc_attr( $subdomain ); ?>" placeholder="<?php esc_attr_e( 'Enter Subdomain (example: mybrand)', 'advanced-form-integration' ); ?>"
                           class="regular-text"/>
                    <p class="description" id="code-description">
                        <?php
                        printf(
                            __( 'Create credentials in <a href="%s" target="_blank" rel="noopener noreferrer">Sendlane API</a> and copy your API key, secret, and subdomain.', 'advanced-form-integration' ),
                            esc_url( 'https://app.sendlane.com/integrations/api' )
                        );
                        ?>
                    </p>
                </td>
            </tr>
        </table>
        <?php submit_button(); ?>
    </form>

    <?php
}

add_action( 'admin_post_adfoin_save_sendlane_credentials', 'adfoin_save_sendlane_credentials', 10, 0 );

function adfoin_save_sendlane_credentials() {
    if ( ! wp_verify_nonce( $_POST['_nonce'], 'adfoin_sendlane_settings' ) ) {
        die( __( 'Security check Failed', 'advanced-form-integration' ) );
    }

    $api_key    = isset( $_POST['adfoin_sendlane_api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['adfoin_sendlane_api_key'] ) ) : '';
    $api_secret = isset( $_POST['adfoin_sendlane_api_secret'] ) ? sanitize_text_field( wp_unslash( $_POST['adfoin_sendlane_api_secret'] ) ) : '';
    $subdomain  = isset( $_POST['adfoin_sendlane_subdomain'] ) ? sanitize_text_field( wp_unslash( $_POST['adfoin_sendlane_subdomain'] ) ) : '';

    update_option( 'adfoin_sendlane_api_key', $api_key );
    update_option( 'adfoin_sendlane_api_secret', $api_secret );
    update_option( 'adfoin_sendlane_subdomain', $subdomain );

    advanced_form_integration_redirect( 'admin.php?page=advanced-form-integration-settings&tab=sendlane' );
}

add_action( 'adfoin_add_js_fields', 'adfoin_sendlane_js_fields', 10, 1 );

function adfoin_sendlane_js_fields( $field_data ) { }

add_action( 'adfoin_action_fields', 'adfoin_sendlane_action_fields' );

function adfoin_sendlane_action_fields() {
    ?>
    <script type="text/template" id="sendlane-action-template">
        <table class="form-table">
            <tr valign="top" v-if="action.task == 'subscribe' || action.task == 'unsubscribe'">
                <th scope="row">
                    <?php esc_attr_e( 'Map Fields', 'advanced-form-integration' ); ?>
                </th>
                <td scope="row">

                </td>
            </tr>

            <tr valign="top" class="alternate" v-if="action.task == 'subscribe' || action.task == 'unsubscribe'">
                <td scope="row-title">
                    <label for="tablecell">
                        <?php esc_attr_e( 'Sendlane List', 'advanced-form-integration' ); ?>
                    </label>
                </td>
                <td>
                    <select name="fieldData[listId]" v-model="fielddata.listId" required="required">
                        <option value=""><?php _e( 'Select List...', 'advanced-form-integration' ); ?></option>
                        <option v-for="(item, index) in fielddata.list" :value="index">{{ item }}</option>
                    </select>
                    <div class="spinner" v-bind:class="{'is-active': listLoading}" style="float:none;width:auto;height:auto;padding:10px 0 10px 50px;background-position:20px 0;"></div>
                </td>
            </tr>

            <editable-field v-for="field in fields"
                            v-bind:key="field.value"
                            v-bind:field="field"
                            v-bind:trigger="trigger"
                            v-bind:action="action"
                            v-bind:fielddata="fielddata"></editable-field>
            <?php
            if ( adfoin_fs()->is__premium_only() && adfoin_fs()->is_plan( 'professional', true ) ) {
                ?>
                <tr valign="top" v-if="action.task == 'subscribe'">
                    <th scope="row">
                        <?php esc_attr_e( 'Using Pro Features', 'advanced-form-integration' ); ?>
                    </th>
                    <td scope="row">
                        <span><?php printf( __( 'For tags, automation triggers, and custom fields, create a <a href="%s">new integration</a> and select Sendlane [PRO].', 'advanced-form-integration' ), admin_url( 'admin.php?page=advanced-form-integration-new' ) ); ?></span>
                    </td>
                </tr>
                <?php
            }

            if ( adfoin_fs()->is_not_paying() ) {
                ?>
                <tr valign="top" v-if="action.task == 'subscribe'">
                    <th scope="row">
                        <?php esc_attr_e( 'Go Pro', 'advanced-form-integration' ); ?>
                    </th>
                    <td scope="row">
                        <span><?php printf( __( 'Unlock tags and automations by <a href="%s">upgrading to Pro</a>.', 'advanced-form-integration' ), admin_url( 'admin.php?page=advanced-form-integration-settings-pricing' ) ); ?></span>
                    </td>
                </tr>
                <?php
            }
            ?>
        </table>
    </script>
    <?php
}

add_action( 'wp_ajax_adfoin_get_sendlane_lists', 'adfoin_get_sendlane_lists', 10, 0 );

function adfoin_get_sendlane_lists() {
    if ( ! wp_verify_nonce( $_POST['_nonce'], 'advanced-form-integration' ) ) {
        die( __( 'Security check Failed', 'advanced-form-integration' ) );
    }

    $lists = array();
    $page  = 1;
    $limit = 50;
    $safe  = 0;

    do {
        $safe++;
        $endpoint = sprintf( 'contacts/lists?page=%d&limit=%d', $page, $limit );
        $response = adfoin_sendlane_request( $endpoint, 'GET' );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error();
        }

        if ( 200 !== wp_remote_retrieve_response_code( $response ) ) {
            wp_send_json_error();
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['data'] ) && is_array( $body['data'] ) ) {
            foreach ( $body['data'] as $list ) {
                if ( isset( $list['id'], $list['name'] ) ) {
                    $lists[ $list['id'] ] = $list['name'];
                }
            }
        }

        if ( isset( $body['meta']['pagination']['next_page'] ) && $body['meta']['pagination']['next_page'] ) {
            $page = (int) $body['meta']['pagination']['next_page'];
        } else {
            $page = 0;
        }

    } while ( $page && $safe < 10 );

    wp_send_json_success( $lists );
}

function adfoin_sendlane_request( $endpoint, $method = 'GET', $data = array(), $record = array() ) {
    $api_key    = get_option( 'adfoin_sendlane_api_key', '' );
    $api_secret = get_option( 'adfoin_sendlane_api_secret', '' );
    $subdomain  = get_option( 'adfoin_sendlane_subdomain', '' );

    if ( ! $api_key || ! $api_secret || ! $subdomain ) {
        return new WP_Error( 'adfoin_sendlane_missing_credentials', __( 'Sendlane credentials are missing.', 'advanced-form-integration' ) );
    }

    $endpoint = ltrim( $endpoint, '/' );
    $url      = sprintf( 'https://%s.sendlane.com/api/v1/%s', $subdomain, $endpoint );

    $args = array(
        'timeout' => 30,
        'method'  => $method,
        'headers' => array(
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
            'API-KEY'       => $api_key,
            'API-SECRET'    => $api_secret,
            'API-ID'        => $subdomain,
        ),
    );

    if ( in_array( strtoupper( $method ), array( 'POST', 'PUT', 'PATCH' ), true ) && ! empty( $data ) ) {
        $args['body'] = wp_json_encode( $data );
    }

    $response = wp_remote_request( $url, $args );

    if ( $record ) {
        adfoin_add_to_log( $response, $url, $args, $record );
    }

    return $response;
}

add_action( 'adfoin_sendlane_job_queue', 'adfoin_sendlane_job_queue', 10, 1 );

function adfoin_sendlane_job_queue( $data ) {
    adfoin_sendlane_send_data( $data['record'], $data['posted_data'] );
}

function adfoin_sendlane_send_data( $record, $posted_data ) {
    $record_data = json_decode( $record['data'], true );

    if ( isset( $record_data['action_data']['cl'] ) && 'yes' === $record_data['action_data']['cl']['active'] ) {
        if ( ! adfoin_match_conditional_logic( $record_data['action_data']['cl'], $posted_data ) ) {
            return;
        }
    }

    $field_data = isset( $record_data['field_data'] ) ? $record_data['field_data'] : array();
    $task       = isset( $record['task'] ) ? $record['task'] : '';

    $list_id = isset( $field_data['listId'] ) ? $field_data['listId'] : '';
    $email   = empty( $field_data['email'] ) ? '' : trim( adfoin_get_parsed_values( $field_data['email'], $posted_data ) );
    $first   = empty( $field_data['firstName'] ) ? '' : adfoin_get_parsed_values( $field_data['firstName'], $posted_data );
    $last    = empty( $field_data['lastName'] ) ? '' : adfoin_get_parsed_values( $field_data['lastName'], $posted_data );

    if ( ! $list_id || ! $email ) {
        return;
    }

    if ( 'unsubscribe' === $task ) {
        $payload = array(
            'emails' => array( $email ),
        );

        adfoin_sendlane_request( sprintf( 'contacts/lists/%s/unsubscribe', $list_id ), 'POST', $payload, $record );

        return;
    }

    if ( 'subscribe' !== $task ) {
        return;
    }

    $payload = array(
        'emails' => array(
            array_filter( array(
                'email'      => $email,
                'first_name' => $first,
                'last_name'  => $last,
            ) ),
        ),
    );

    adfoin_sendlane_request( sprintf( 'contacts/lists/%s/subscribe', $list_id ), 'POST', $payload, $record );
}
