<?php

add_filter( 'adfoin_action_providers', 'adfoin_phplist_actions', 10, 1 );

function adfoin_phplist_actions( $actions ) {

    $actions['phplist'] = array(
        'title' => __( 'phpList', 'advanced-form-integration' ),
        'tasks' => array(
            'add_to_list' => __( 'Create / Subscribe Subscriber', 'advanced-form-integration' ),
        ),
    );

    return $actions;
}

add_filter( 'adfoin_settings_tabs', 'adfoin_phplist_settings_tab', 10, 1 );

function adfoin_phplist_settings_tab( $tabs ) {
    $tabs['phplist'] = __( 'phpList', 'advanced-form-integration' );

    return $tabs;
}

add_action( 'adfoin_settings_view', 'adfoin_phplist_settings_view', 10, 1 );

function adfoin_phplist_settings_view( $current_tab ) {
    if ( 'phplist' !== $current_tab ) {
        return;
    }

    $title      = __( 'phpList', 'advanced-form-integration' );
    $key        = 'phplist';
    $arguments  = wp_json_encode(
        array(
            'platform' => $key,
            'fields'   => array(
                array(
                    'key'   => 'baseUrl',
                    'label' => __( 'API Base URL (ends with /api/v2)', 'advanced-form-integration' ),
                    'hidden' => false,
                ),
                array(
                    'key'   => 'username',
                    'label' => __( 'Admin Username', 'advanced-form-integration' ),
                    'hidden' => false,
                ),
                array(
                    'key'   => 'password',
                    'label' => __( 'Admin Password', 'advanced-form-integration' ),
                    'hidden' => true,
                ),
            ),
        )
    );
    $instructions = sprintf(
        /* translators: 1: opening anchor tag, 2: closing anchor tag */
        __( '<p>Enable the phpList REST API (see %1$sAPI documentation%2$s), then enter the full API base URL (e.g. https://example.com/lists/api/v2) and the admin credentials that have API access. AFI will create a session token automatically for every request.</p>', 'advanced-form-integration' ),
        '<a href="https://www.phplist.org/manual/books/phplist-manual/page/api-and-integrations" target="_blank" rel="noopener noreferrer">',
        '</a>'
    );

    echo adfoin_platform_settings_template( $title, $key, $arguments, $instructions ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
}

add_action( 'wp_ajax_adfoin_get_phplist_credentials', 'adfoin_get_phplist_credentials', 10, 0 );

function adfoin_get_phplist_credentials() {
    if ( ! adfoin_verify_nonce() ) {
        return;
    }

    $credentials = adfoin_read_credentials( 'phplist' );

    wp_send_json_success( $credentials );
}

add_action( 'wp_ajax_adfoin_save_phplist_credentials', 'adfoin_save_phplist_credentials', 10, 0 );

function adfoin_save_phplist_credentials() {
    if ( ! adfoin_verify_nonce() ) {
        return;
    }

    $platform = isset( $_POST['platform'] ) ? sanitize_text_field( wp_unslash( $_POST['platform'] ) ) : '';

    if ( 'phplist' === $platform ) {
        $data = isset( $_POST['data'] ) ? adfoin_array_map_recursive( 'sanitize_text_field', wp_unslash( $_POST['data'] ) ) : array();

        adfoin_save_credentials( $platform, $data );
    }

    wp_send_json_success();
}

function adfoin_phplist_credentials_list() {
    $html        = '';
    $credentials = adfoin_read_credentials( 'phplist' );

    foreach ( $credentials as $credential ) {
        $title   = isset( $credential['title'] ) ? $credential['title'] : '';
        $html   .= '<option value="' . esc_attr( $credential['id'] ) . '">' . esc_html( $title ) . '</option>';
    }

    echo $html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
}

add_action( 'adfoin_action_fields', 'adfoin_phplist_action_fields' );

function adfoin_phplist_action_fields() {
    ?>
    <script type="text/template" id="phplist-action-template">
        <table class="form-table" v-if="action.task == 'add_to_list'">
            <tr valign="top">
                <th scope="row">
                    <?php esc_html_e( 'Map Fields', 'advanced-form-integration' ); ?>
                </th>
                <td></td>
            </tr>
            <tr class="alternate">
                <td scope="row-title">
                    <label><?php esc_html_e( 'phpList Account', 'advanced-form-integration' ); ?></label>
                </td>
                <td>
                    <select name="fieldData[credId]" v-model="fielddata.credId">
                        <option value=""><?php esc_html_e( 'Select credentials…', 'advanced-form-integration' ); ?></option>
                        <?php adfoin_phplist_credentials_list(); ?>
                    </select>
                </td>
            </tr>
            <tr class="alternate">
                <td scope="row-title">
                    <label><?php esc_html_e( 'Subscriber List', 'advanced-form-integration' ); ?></label>
                </td>
                <td>
                    <select name="fieldData[listId]" v-model="fielddata.listId">
                        <option value=""><?php esc_html_e( 'Select list…', 'advanced-form-integration' ); ?></option>
                        <option v-for="(label, value) in fielddata.lists" :value="value">{{ label }}</option>
                    </select>
                    <div class="spinner" v-bind:class="{'is-active': listLoading}" style="float:none;width:auto;height:auto;padding:10px 0 10px 50px;background-position:20px 0;"></div>
                </td>
            </tr>
            <tr class="alternate">
                <td scope="row-title">
                    <label><?php esc_html_e( 'Options', 'advanced-form-integration' ); ?></label>
                </td>
                <td>
                    <label style="margin-right: 15px;">
                        <input type="checkbox" name="fieldData[requestConfirmation]" value="true" v-model="fielddata.requestConfirmation">
                        <?php esc_html_e( 'Request double opt-in confirmation', 'advanced-form-integration' ); ?>
                    </label>
                    <label>
                        <input type="checkbox" name="fieldData[htmlEmail]" value="true" v-model="fielddata.htmlEmail">
                        <?php esc_html_e( 'Send HTML email', 'advanced-form-integration' ); ?>
                    </label>
                </td>
            </tr>
            <tr class="alternate">
                <td scope="row-title">
                    <label><?php esc_html_e( 'Instructions', 'advanced-form-integration' ); ?></label>
                </td>
                <td>
                    <a href="https://www.phplist.org/manual/books/phplist-manual/page/api-and-integrations" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'API documentation', 'advanced-form-integration' ); ?></a>
                </td>
            </tr>
            <editable-field v-for="field in fields"
                v-bind:key="field.value"
                v-bind:field="field"
                v-bind:trigger="trigger"
                v-bind:action="action"
                v-bind:fielddata="fielddata"></editable-field>
        </table>
    </script>
    <?php
}

add_action( 'wp_ajax_adfoin_get_phplist_lists', 'adfoin_get_phplist_lists', 10, 0 );

function adfoin_get_phplist_lists() {
    if ( ! adfoin_verify_nonce() ) {
        return;
    }

    $cred_id = isset( $_POST['credId'] ) ? sanitize_text_field( wp_unslash( $_POST['credId'] ) ) : '';

    if ( empty( $cred_id ) ) {
        wp_send_json_error(
            array(
                'message' => __( 'Select a phpList credential first.', 'advanced-form-integration' ),
            )
        );
    }

    $credentials = adfoin_get_credentials_by_id( 'phplist', $cred_id );

    if ( empty( $credentials ) ) {
        wp_send_json_error(
            array(
                'message' => __( 'Invalid phpList credential.', 'advanced-form-integration' ),
            )
        );
    }

    $lists = adfoin_phplist_fetch_lists( $credentials );

    if ( is_wp_error( $lists ) ) {
        wp_send_json_error(
            array(
                'message' => $lists->get_error_message(),
            )
        );
    }

    wp_send_json_success( $lists );
}

add_action( 'adfoin_phplist_job_queue', 'adfoin_phplist_job_queue', 10, 1 );

function adfoin_phplist_job_queue( $data ) {
    adfoin_phplist_send_data( $data['record'], $data['posted_data'] );
}

function adfoin_phplist_send_data( $record, $posted_data ) {
    $record_data = json_decode( $record['data'], true );

    if ( isset( $record_data['action_data']['cl'] ) && isset( $record_data['action_data']['cl']['active'] ) ) {
        if ( 'yes' === $record_data['action_data']['cl']['active'] ) {
            if ( ! adfoin_match_conditional_logic( $record_data['action_data']['cl'], $posted_data ) ) {
                return;
            }
        }
    }

    $field_data = isset( $record_data['field_data'] ) ? $record_data['field_data'] : array();
    $cred_id    = isset( $field_data['credId'] ) ? $field_data['credId'] : '';

    $email_value = isset( $field_data['email'] ) ? $field_data['email'] : '';
    $email       = $email_value ? adfoin_get_parsed_values( $email_value, $posted_data ) : '';
    $email       = sanitize_email( $email );

    if ( empty( $cred_id ) || empty( $email ) ) {
        return;
    }

    $credentials = adfoin_get_credentials_by_id( 'phplist', $cred_id );

    if ( empty( $credentials ) ) {
        return;
    }

    $list_id              = isset( $field_data['listId'] ) ? absint( $field_data['listId'] ) : 0;
    $request_confirmation = isset( $field_data['requestConfirmation'] ) && 'true' === $field_data['requestConfirmation'];
    $html_email           = ! isset( $field_data['htmlEmail'] ) || 'true' === $field_data['htmlEmail'];

    $session  = null;
    $payload  = array( 'email' => $email );

    if ( $request_confirmation ) {
        $payload['requestConfirmation'] = true;
    }

    if ( isset( $field_data['htmlEmail'] ) ) {
        $payload['htmlEmail'] = $html_email;
    }

    $response = adfoin_phplist_api_request( $credentials, 'subscribers', 'POST', $payload, $record, $session );

    if ( is_wp_error( $response ) ) {
        return;
    }

    $status = wp_remote_retrieve_response_code( $response );

    if ( ! in_array( (int) $status, array( 201, 409 ), true ) ) {
        return;
    }

    if ( $list_id > 0 ) {
        $subscribe_payload = array(
            'emails' => array( $email ),
        );

        $endpoint = sprintf( 'lists/%d/subscribers', $list_id );
        $response = adfoin_phplist_api_request( $credentials, $endpoint, 'POST', $subscribe_payload, $record, $session );

        if ( is_wp_error( $response ) ) {
            return;
        }

        $status = wp_remote_retrieve_response_code( $response );

        if ( ! in_array( (int) $status, array( 200, 201, 204, 409 ), true ) ) {
            return;
        }
    }
}

function adfoin_phplist_fetch_lists( $credentials ) {
    $lists   = array();
    $session = null;
    $cursor  = 0;
    $fetch   = true;

    while ( $fetch ) {
        $endpoint = 'lists?limit=100';

        if ( $cursor > 0 ) {
            $endpoint .= '&after_id=' . $cursor;
        }

        $response = adfoin_phplist_api_request( $credentials, $endpoint, 'GET', array(), array(), $session );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $status = wp_remote_retrieve_response_code( $response );

        if ( 200 !== (int) $status ) {
            return new WP_Error( 'phplist_list_error', adfoin_phplist_extract_error_message( $response ) );
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['items'] ) && is_array( $body['items'] ) ) {
            foreach ( $body['items'] as $item ) {
                if ( isset( $item['id'], $item['name'] ) ) {
                    $lists[ $item['id'] ] = $item['name'];
                    $cursor               = (int) $item['id'];
                }
            }
        }

        $has_more = isset( $body['pagination']['has_more'] ) ? (bool) $body['pagination']['has_more'] : false;

        if ( $has_more && isset( $body['pagination']['next_cursor'] ) ) {
            $cursor = (int) $body['pagination']['next_cursor'];
        } else {
            $fetch = false;
        }
    }

    return $lists;
}

function adfoin_phplist_api_request( $credentials, $endpoint, $method = 'GET', $payload = array(), $record = array(), &$session = null ) {
    $base_url = adfoin_phplist_normalize_base_url( $credentials );

    if ( is_wp_error( $base_url ) ) {
        return $base_url;
    }

    if ( null === $session ) {
        $session = adfoin_phplist_create_session( $credentials, $base_url );

        if ( is_wp_error( $session ) ) {
            return $session;
        }
    }

    $url = trailingslashit( $base_url ) . ltrim( $endpoint, '/' );

    $headers = array(
        'Content-Type'  => 'application/json',
        'php-auth-pw'   => $session['token'],
        'php-auth-user' => $credentials['username'],
        'Authorization' => 'Basic ' . base64_encode( $credentials['username'] . ':' . $session['token'] ), // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions
    );

    $args = array(
        'timeout' => 30,
        'method'  => $method,
        'headers' => $headers,
    );

    if ( in_array( strtoupper( $method ), array( 'POST', 'PUT', 'PATCH' ), true ) ) {
        $args['body'] = wp_json_encode( $payload );
    }

    $response = wp_remote_request( $url, $args );

    if ( $record ) {
        adfoin_add_to_log( $response, $url, $args, $record );
    }

    return $response;
}

function adfoin_phplist_create_session( $credentials, $base_url ) {
    $username = isset( $credentials['username'] ) ? $credentials['username'] : '';
    $password = isset( $credentials['password'] ) ? $credentials['password'] : '';

    if ( empty( $username ) || empty( $password ) ) {
        return new WP_Error( 'phplist_missing_credentials', __( 'phpList username or password is missing.', 'advanced-form-integration' ) );
    }

    $url = trailingslashit( $base_url ) . 'sessions';

    $args = array(
        'timeout' => 30,
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body'    => wp_json_encode(
            array(
                'login_name' => $username,
                'password'   => $password,
            )
        ),
    );

    $response = wp_remote_post( $url, $args );

    if ( is_wp_error( $response ) ) {
        return $response;
    }

    $status = wp_remote_retrieve_response_code( $response );

    if ( 201 !== (int) $status ) {
        return new WP_Error( 'phplist_login_failed', adfoin_phplist_extract_error_message( $response ) );
    }

    $body = json_decode( wp_remote_retrieve_body( $response ), true );

    if ( empty( $body['key'] ) ) {
        return new WP_Error( 'phplist_login_failed', __( 'phpList session token missing from response.', 'advanced-form-integration' ) );
    }

    return array(
        'token' => $body['key'],
        'id'    => isset( $body['id'] ) ? $body['id'] : null,
    );
}

function adfoin_phplist_normalize_base_url( $credentials ) {
    $base_url = isset( $credentials['baseUrl'] ) ? trim( $credentials['baseUrl'] ) : '';

    if ( empty( $base_url ) ) {
        return new WP_Error( 'phplist_missing_baseurl', __( 'phpList API Base URL is missing.', 'advanced-form-integration' ) );
    }

    if ( ! preg_match( '#^https?://#i', $base_url ) ) {
        $base_url = 'https://' . $base_url;
    }

    return untrailingslashit( $base_url );
}

function adfoin_phplist_extract_error_message( $response ) {
    if ( is_wp_error( $response ) ) {
        return $response->get_error_message();
    }

    $body = json_decode( wp_remote_retrieve_body( $response ), true );

    if ( isset( $body['message'] ) ) {
        return $body['message'];
    }

    if ( isset( $body['detail'] ) ) {
        return $body['detail'];
    }

    return __( 'Unexpected phpList API error.', 'advanced-form-integration' );
}
