<?php

class ADFOIN_GoogleTasks extends Advanced_Form_Integration_OAuth2 {

    const service_name           = 'googletasks';
    const authorization_endpoint = 'https://accounts.google.com/o/oauth2/auth';
    const token_endpoint         = 'https://www.googleapis.com/oauth2/v3/token';

    private static $instance;
    protected $client_id      = '';
    protected $client_secret  = '';
    protected $task_lists     = array();

    public static function get_instance() {
        if ( empty( self::$instance ) ) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    private function __construct() {
        $this->token_endpoint         = self::token_endpoint;
        $this->authorization_endpoint = self::authorization_endpoint;

        $option = (array) maybe_unserialize( get_option( 'adfoin_googletasks_keys' ) );

        if ( isset( $option['client_id'] ) ) {
            $this->client_id = $option['client_id'];
        }

        if ( isset( $option['client_secret'] ) ) {
            $this->client_secret = $option['client_secret'];
        }

        if ( isset( $option['access_token'] ) ) {
            $this->access_token = $option['access_token'];
        }

        if ( isset( $option['refresh_token'] ) ) {
            $this->refresh_token = $option['refresh_token'];
        }

        if ( $this->is_active() && isset( $option['task_lists'] ) ) {
            $this->task_lists = $option['task_lists'];
        }

        add_action( 'admin_init', array( $this, 'auth_redirect' ) );
        add_action( 'rest_api_init', array( $this, 'register_callback_route' ) );
        add_filter( 'adfoin_action_providers', array( $this, 'register_actions' ), 10, 1 );
        add_filter( 'adfoin_settings_tabs', array( $this, 'register_settings_tab' ), 10, 1 );
        add_action( 'adfoin_settings_view', array( $this, 'render_settings' ), 10, 1 );
        add_action( 'adfoin_action_fields', array( $this, 'action_fields' ), 10, 1 );
        add_action( 'admin_post_adfoin_save_googletasks_keys', array( $this, 'save_keys' ), 10, 0 );
        add_action( 'wp_ajax_adfoin_get_googletasks_lists', array( $this, 'ajax_get_lists' ), 10, 0 );
    }

    public function register_callback_route() {
        register_rest_route(
            'advancedformintegration',
            '/googletasks',
            array(
                'methods'             => 'GET',
                'callback'            => array( $this, 'handle_webhook' ),
                'permission_callback' => '__return_true',
            )
        );
    }

    public function handle_webhook( $request ) {
        $params = $request->get_params();
        $code   = isset( $params['code'] ) ? trim( $params['code'] ) : '';

        if ( $code ) {
            $redirect_to = add_query_arg(
                array(
                    'service' => 'authorize',
                    'action'  => 'adfoin_googletasks_auth_redirect',
                    'code'    => $code,
                ),
                admin_url( 'admin.php?page=advanced-form-integration' )
            );

            wp_safe_redirect( $redirect_to );
            exit();
        }
    }

    public function auth_redirect() {
        $action = isset( $_GET['action'] ) ? sanitize_text_field( trim( $_GET['action'] ) ) : '';

        if ( 'adfoin_googletasks_auth_redirect' === $action ) {
            $code = isset( $_GET['code'] ) ? sanitize_text_field( $_GET['code'] ) : '';

            if ( $code ) {
                $this->request_token( $code );
            }

            wp_safe_redirect( admin_url( 'admin.php?page=advanced-form-integration-settings&tab=googletasks' ) );
            exit();
        }
    }

    public function register_actions( $actions ) {
        $actions['googletasks'] = array(
            'title' => __( 'Google Tasks', 'advanced-form-integration' ),
            'tasks' => array(
                'create_task' => __( 'Create Task', 'advanced-form-integration' ),
            ),
        );

        return $actions;
    }

    public function register_settings_tab( $tabs ) {
        $tabs['googletasks'] = __( 'Google Tasks', 'advanced-form-integration' );
        return $tabs;
    }

    public function render_settings( $current_tab ) {
        if ( 'googletasks' !== $current_tab ) {
            return;
        }

        $option        = (array) maybe_unserialize( get_option( 'adfoin_googletasks_keys' ) );
        $nonce         = wp_create_nonce( 'adfoin_googletasks_settings' );
        $client_id     = isset( $option['client_id'] ) ? $option['client_id'] : '';
        $client_secret = isset( $option['client_secret'] ) ? $option['client_secret'] : '';
        $redirect_uri  = $this->get_redirect_uri();
        $domain        = parse_url( get_site_url() );
        $host          = isset( $domain['host'] ) ? $domain['host'] : '';
        ?>
        <form action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" method="post" class="container">
            <input type="hidden" name="action" value="adfoin_save_googletasks_keys">
            <input type="hidden" name="_nonce" value="<?php echo esc_attr( $nonce ); ?>">

            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><?php esc_html_e( 'Instructions', 'advanced-form-integration' ); ?></th>
                    <td>
                        <p>
                            <?php
                            printf(
                                /* translators: 1: host, 2: redirect URI */
                                __( '1. Open the <a href="https://console.cloud.google.com/project" target="_blank" rel="noopener noreferrer">Google Cloud Console</a> and create (or select) a project.<br>2. Enable the <strong>Google Tasks API</strong> in the API Library.<br>3. Configure the OAuth consent screen (External), add <code>%1$s</code> to Authorized domains, and publish it.<br>4. Under Credentials, create an <strong>OAuth client ID</strong> with type Web Application. Add <code>%2$s</code> as an authorized redirect URI.<br>5. Copy the Client ID and Client Secret below, then click Save &amp; Authorize to grant the <code>https://www.googleapis.com/auth/tasks</code> scope.<br>', 'advanced-form-integration' ),
                                esc_html( $host ),
                                esc_html( $redirect_uri )
                            );
                            ?>
                        </p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e( 'Status', 'advanced-form-integration' ); ?></th>
                    <td>
                        <?php echo isset( $option['refresh_token'] ) && $option['refresh_token'] ? esc_html__( 'Connected', 'advanced-form-integration' ) : esc_html__( 'Not Connected', 'advanced-form-integration' ); ?>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e( 'Client ID', 'advanced-form-integration' ); ?></th>
                    <td>
                        <input type="text" class="regular-text" name="adfoin_googletasks_client_id" value="<?php echo esc_attr( $client_id ); ?>" placeholder="<?php esc_attr_e( 'Enter Client ID', 'advanced-form-integration' ); ?>">
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e( 'Client Secret', 'advanced-form-integration' ); ?></th>
                    <td>
                        <input type="text" class="regular-text" name="adfoin_googletasks_client_secret" value="<?php echo esc_attr( $client_secret ); ?>" placeholder="<?php esc_attr_e( 'Enter Client Secret', 'advanced-form-integration' ); ?>">
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e( 'Re-authorize', 'advanced-form-integration' ); ?></th>
                    <td>
                        <?php
                        _e( 'If Google revokes the connection, remove the previous access from <a target="_blank" rel="noopener noreferrer" href="https://myaccount.google.com/permissions">Google App Permissions</a> and click Save &amp; Authorize again.', 'advanced-form-integration' );
                        ?>
                    </td>
                </tr>
            </table>

            <?php submit_button( __( 'Save & Authorize', 'advanced-form-integration' ) ); ?>
        </form>
        <?php
    }

    public function save_keys() {
        if ( ! isset( $_POST['_nonce'] ) || ! wp_verify_nonce( $_POST['_nonce'], 'adfoin_googletasks_settings' ) ) {
            wp_die( __( 'Security check failed', 'advanced-form-integration' ) );
        }

        $client_id     = isset( $_POST['adfoin_googletasks_client_id'] ) ? sanitize_text_field( $_POST['adfoin_googletasks_client_id'] ) : '';
        $client_secret = isset( $_POST['adfoin_googletasks_client_secret'] ) ? sanitize_text_field( $_POST['adfoin_googletasks_client_secret'] ) : '';

        if ( empty( $client_id ) || empty( $client_secret ) ) {
            $this->reset_data();
            advanced_form_integration_redirect( 'admin.php?page=advanced-form-integration-settings&tab=googletasks' );
            exit;
        }

        $this->client_id     = trim( $client_id );
        $this->client_secret = trim( $client_secret );

        $this->save_data();
        $this->authorize( 'https://www.googleapis.com/auth/tasks' );
    }

    protected function authorize( $scope = '' ) {
        $endpoint = add_query_arg(
            array(
                'response_type' => 'code',
                'access_type'   => 'offline',
                'prompt'        => 'consent',
                'client_id'     => $this->client_id,
                'redirect_uri'  => urlencode( $this->get_redirect_uri() ),
                'scope'         => urlencode( $scope ),
            ),
            $this->authorization_endpoint
        );

        if ( wp_redirect( esc_url_raw( $endpoint ) ) ) {
            exit();
        }
    }

    protected function request_token( $authorization_code ) {
        $args = array(
            'headers' => array(),
            'body'    => array(
                'code'          => $authorization_code,
                'client_id'     => $this->client_id,
                'client_secret' => $this->client_secret,
                'redirect_uri'  => $this->get_redirect_uri(),
                'grant_type'    => 'authorization_code',
                'access_type'   => 'offline',
                'prompt'        => 'consent',
            ),
        );

        $response      = wp_remote_post( esc_url_raw( $this->token_endpoint ), $args );
        $response_body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $response_body['access_token'] ) ) {
            $this->access_token = $response_body['access_token'];
        } else {
            $this->access_token = null;
        }

        if ( isset( $response_body['refresh_token'] ) ) {
            $this->refresh_token = $response_body['refresh_token'];
        } else {
            $this->refresh_token = null;
        }

        $this->save_data();

        return $response;
    }

    protected function refresh_token() {
        $args = array(
            'headers' => array(),
            'body'    => array(
                'refresh_token' => $this->refresh_token,
                'client_id'     => $this->client_id,
                'client_secret' => $this->client_secret,
                'grant_type'    => 'refresh_token',
            ),
        );

        $response      = wp_remote_post( esc_url_raw( $this->token_endpoint ), $args );
        $response_body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $response_body['access_token'] ) ) {
            $this->access_token = $response_body['access_token'];
        } else {
            $this->access_token = null;
        }

        if ( isset( $response_body['refresh_token'] ) ) {
            $this->refresh_token = $response_body['refresh_token'];
        }

        $this->save_data();

        return $response;
    }

    protected function save_data() {
        $data = (array) maybe_unserialize( get_option( 'adfoin_googletasks_keys' ) );

        $option = array_merge(
            $data,
            array(
                'client_id'     => $this->client_id,
                'client_secret' => $this->client_secret,
                'access_token'  => $this->access_token,
                'refresh_token' => $this->refresh_token,
                'task_lists'    => $this->task_lists,
            )
        );

        update_option( 'adfoin_googletasks_keys', maybe_serialize( $option ) );
    }

    protected function reset_data() {
        $this->client_id     = '';
        $this->client_secret = '';
        $this->access_token  = '';
        $this->refresh_token = '';
        $this->task_lists    = array();

        $this->save_data();
    }

    protected function get_redirect_uri() {
        return site_url( '/wp-json/advancedformintegration/googletasks' );
    }

    public function ajax_get_lists() {
        if ( ! adfoin_verify_nonce() ) {
            wp_send_json_error( __( 'Invalid nonce.', 'advanced-form-integration' ) );
        }

        $response = $this->fetch_task_lists();

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( $response->get_error_message() );
        }

        wp_send_json_success( $response );
    }

    protected function fetch_task_lists() {
        if ( ! $this->access_token ) {
            return new WP_Error( 'googletasks_no_token', __( 'Authorize Google Tasks first.', 'advanced-form-integration' ) );
        }

        if ( ! $this->verify_token_active() ) {
            $this->refresh_token();
        }

        $endpoint = 'https://tasks.googleapis.com/tasks/v1/users/@me/lists';
        $request  = array(
            'method'  => 'GET',
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->access_token,
            ),
        );

        $response = $this->remote_request( $endpoint, $request );
        $body     = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( isset( $body['items'] ) && is_array( $body['items'] ) ) {
            $lists = array();
            foreach ( $body['items'] as $item ) {
                if ( isset( $item['id'], $item['title'] ) ) {
                    $lists[ $item['id'] ] = $item['title'];
                }
            }

            $this->task_lists = $lists;
            $this->save_data();

            return $lists;
        }

        return array();
    }

    protected function verify_token_active() {
        if ( empty( $this->access_token ) ) {
            return false;
        }

        $response = wp_remote_get( 'https://www.googleapis.com/oauth2/v1/tokeninfo?access_token=' . $this->access_token );

        if ( is_wp_error( $response ) ) {
            return false;
        }

        return 200 === wp_remote_retrieve_response_code( $response );
    }

    public function action_fields() {
        ?>
        <script type="text/template" id="googletasks-action-template">
            <table class="form-table" v-if="action.task == 'create_task'">
                <tr valign="top">
                    <th scope="row"><?php esc_html_e( 'Map Fields', 'advanced-form-integration' ); ?></th>
                    <td></td>
                </tr>

                <tr valign="top" class="alternate">
                    <td scope="row-title">
                        <label><?php esc_html_e( 'Task List', 'advanced-form-integration' ); ?></label>
                    </td>
                    <td>
                        <select name="fieldData[listId]" v-model="fielddata.listId" required="required">
                            <option value=""><?php esc_html_e( 'Select list…', 'advanced-form-integration' ); ?></option>
                            <option v-for="(label, id) in fielddata.taskLists" :value="id">{{ label }}</option>
                        </select>
                        <div class="spinner" v-bind:class="{'is-active': listsLoading}" style="float:none;width:auto;height:auto;padding:10px 0 10px 50px;background-position:20px 0;"></div>
                    </td>
                </tr>

                <tr valign="top" class="alternate">
                    <td scope="row-title">
                        <label><?php esc_html_e( 'Docs', 'advanced-form-integration' ); ?></label>
                    </td>
                    <td>
                        <a href="https://developers.google.com/tasks/reference/rest" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Google Tasks REST reference', 'advanced-form-integration' ); ?></a>
                    </td>
                </tr>

                <editable-field v-for="field in fields"
                    v-bind:key="field.value"
                    v-bind:field="field"
                    v-bind:trigger="trigger"
                    v-bind:action="action"
                    v-bind:fielddata="fielddata"></editable-field>
            </table>
        </script>
        <?php
    }

    public function create_task( $list_id, $task_data, $record ) {
        if ( ! $list_id || empty( $task_data ) ) {
            return;
        }

        if ( ! $this->verify_token_active() ) {
            $this->refresh_token();
        }

        $endpoint = sprintf( 'https://tasks.googleapis.com/tasks/v1/lists/%s/tasks', rawurlencode( $list_id ) );
        $request  = array(
            'method'  => 'POST',
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->access_token,
                'Content-Type'  => 'application/json',
            ),
            'body'    => wp_json_encode( $task_data ),
        );

        $response = $this->remote_request( $endpoint, $request );
        adfoin_add_to_log( $response, $endpoint, $request, $record );
    }
}

$googletasks = ADFOIN_GoogleTasks::get_instance();

add_action( 'adfoin_googletasks_job_queue', 'adfoin_googletasks_job_queue', 10, 1 );

function adfoin_googletasks_job_queue( $data ) {
    adfoin_googletasks_send_data( $data['record'], $data['posted_data'] );
}

function adfoin_googletasks_send_data( $record, $posted_data ) {
    $record_data = json_decode( $record['data'], true );

    if ( adfoin_check_conditional_logic( $record_data['action_data']['cl'] ?? array(), $posted_data ) ) {
        return;
    }

    $data = isset( $record_data['field_data'] ) ? $record_data['field_data'] : array();
    $task = isset( $record['task'] ) ? $record['task'] : '';

    if ( 'create_task' !== $task ) {
        return;
    }

    $list_id = isset( $data['listId'] ) ? $data['listId'] : '';
    $title   = empty( $data['title'] ) ? '' : adfoin_get_parsed_values( $data['title'], $posted_data );

    if ( empty( $list_id ) || empty( $title ) ) {
        return;
    }

    $notes     = empty( $data['notes'] ) ? '' : adfoin_get_parsed_values( $data['notes'], $posted_data );
    $due       = empty( $data['due'] ) ? '' : adfoin_get_parsed_values( $data['due'], $posted_data );
    $status    = empty( $data['status'] ) ? '' : strtolower( adfoin_get_parsed_values( $data['status'], $posted_data ) );
    $parent    = empty( $data['parent'] ) ? '' : adfoin_get_parsed_values( $data['parent'], $posted_data );
    $position  = empty( $data['position'] ) ? '' : adfoin_get_parsed_values( $data['position'], $posted_data );

    $payload = array( 'title' => $title );

    if ( $notes ) {
        $payload['notes'] = $notes;
    }

    if ( $due ) {
        $timestamp = strtotime( $due );
        if ( false !== $timestamp ) {
            $payload['due'] = gmdate( 'Y-m-d\TH:i:s.000\Z', $timestamp );
        }
    }

    if ( $parent ) {
        $payload['parent'] = $parent;
    }

    if ( $position ) {
        $payload['position'] = $position;
    }

    if ( in_array( $status, array( 'completed', 'needsaction' ), true ) ) {
        $payload['status'] = ( 'completed' === $status ) ? 'completed' : 'needsAction';
        if ( 'completed' === $payload['status'] ) {
            $payload['completed'] = gmdate( 'Y-m-d\TH:i:s.000\Z' );
        }
    }

    $googletasks = ADFOIN_GoogleTasks::get_instance();
    $googletasks->create_task( $list_id, $payload, $record );
}
