<?php

namespace Org\Wplake\Advanced_Views\Optional_Vendors\Illuminate\Support;

use Org\Wplake\Advanced_Views\Optional_Vendors\Dotenv\Repository\Adapter\PutenvAdapter;
use Org\Wplake\Advanced_Views\Optional_Vendors\Dotenv\Repository\RepositoryBuilder;
use Org\Wplake\Advanced_Views\Optional_Vendors\PhpOption\Option;
use RuntimeException;
class Env
{
    /**
     * Indicates if the putenv adapter is enabled.
     *
     * @var bool
     */
    protected static $putenv = \true;
    /**
     * The environment repository instance.
     *
     * @var \Dotenv\Repository\RepositoryInterface|null
     */
    protected static $repository;
    /**
     * Enable the putenv adapter.
     *
     * @return void
     */
    public static function enablePutenv()
    {
        static::$putenv = \true;
        static::$repository = null;
    }
    /**
     * Disable the putenv adapter.
     *
     * @return void
     */
    public static function disablePutenv()
    {
        static::$putenv = \false;
        static::$repository = null;
    }
    /**
     * Get the environment repository instance.
     *
     * @return \Dotenv\Repository\RepositoryInterface
     */
    public static function getRepository()
    {
        if (static::$repository === null) {
            $builder = RepositoryBuilder::createWithDefaultAdapters();
            if (static::$putenv) {
                $builder = $builder->addAdapter(PutenvAdapter::class);
            }
            static::$repository = $builder->immutable()->make();
        }
        return static::$repository;
    }
    /**
     * Get the value of an environment variable.
     *
     * @param  string  $key
     * @param  mixed  $default
     * @return mixed
     */
    public static function get($key, $default = null)
    {
        return self::getOption($key)->getOrCall(fn() => value($default));
    }
    /**
     * Get the value of a required environment variable.
     *
     * @param  string  $key
     * @return mixed
     *
     * @throws \RuntimeException
     */
    public static function getOrFail($key)
    {
        return self::getOption($key)->getOrThrow(new RuntimeException("Environment variable [{$key}] has no value."));
    }
    /**
     * Get the possible option for this environment variable.
     *
     * @param  string  $key
     * @return \PhpOption\Option|\PhpOption\Some
     */
    protected static function getOption($key)
    {
        return Option::fromValue(static::getRepository()->get($key))->map(function ($value) {
            switch (\strtolower($value)) {
                case 'true':
                case '(true)':
                    return \true;
                case 'false':
                case '(false)':
                    return \false;
                case 'empty':
                case '(empty)':
                    return '';
                case 'null':
                case '(null)':
                    return;
            }
            if (\preg_match('/\\A([\'"])(.*)\\1\\z/', $value, $matches)) {
                return $matches[2];
            }
            return $value;
        });
    }
}
