<?php
/**
 * Returns an array of WCAG Success Criteria and relevant information.
 *
 * @package Accessibility_Checker
 */

return [
	[
		'number'               => '1.1.1',
		'title'                => esc_html__( 'Non-text Content', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'All non-text content must have a text alternative that serves the same purpose, unless it falls into specific exceptions like controls, media, tests, sensory content, CAPTCHA, or decorative elements. Each type must follow specific guidance to ensure accessibility without interfering with the user experience.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.1 Text Alternatives', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with dexterity impairments using voice control', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#non-text-content',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/non-text-content.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#non-text-content',
		'tags'                 => [ 'img', 'svg', 'icon', 'input', 'button', 'media', 'captcha', 'decorative', 'text alternative', 'alt' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.2.1',
		'title'                => esc_html__( 'Prerecorded Audio-only and Video-only', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For prerecorded audio-only and video-only content, a text alternative must be provided that presents equivalent information. This ensures users who cannot hear or see the media can still access its content through text.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#audio-only-and-video-only-prerecorded',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/audio-only-and-video-only-prerecorded.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#audio-only-and-video-only-prerecorded',
		'tags'                 => [ 'audio', 'video', 'media', 'transcript', 'text alternative' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.2.2',
		'title'                => esc_html__( 'Captions (Prerecorded)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Captions must be provided for all prerecorded audio content in synchronized media—such as videos where audio and visuals play together—so that users who are deaf or hard of hearing can access the spoken information and important sounds.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#captions-prerecorded',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/captions-prerecorded.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#captions-prerecorded',
		'tags'                 => [ 'video', 'audio', 'media', 'captions', 'subtitles', 'transcript' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.2.3',
		'title'                => esc_html__( 'Audio Description or Media Alternative (Prerecorded)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For prerecorded video content with audio (synchronized media), an audio description—or a written transcript that includes visual details—must be provided to ensure users who are blind or have low vision can understand important visual information that isn\'t spoken.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#audio-description-or-media-alternative-prerecorded',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/audio-description-or-media-alternative-prerecorded.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#audio-description-or-media-alternative-prerecorded',
		'tags'                 => [ 'video', 'audio', 'media', 'audio description', 'transcript', 'description' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.2.4',
		'title'                => esc_html__( 'Captions (Live)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For all live video content that includes audio, captions must be provided in real time so that users who are deaf or hard of hearing can access the spoken information as it happens.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#captions-live',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/captions-live.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#captions-live',
		'tags'                 => [ 'video', 'audio', 'media', 'captions', 'live', 'real-time' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.2.5',
		'title'                => esc_html__( 'Audio Description (Prerecorded)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For prerecorded video with audio, an audio description of important visual content must be provided, allowing users who are blind or have low vision to understand visual information that is not described in the main audio.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#audio-description-prerecorded',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/audio-description-prerecorded.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#audio-description-prerecorded',
		'tags'                 => [ 'video', 'audio', 'media', 'audio description', 'description' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.2.6',
		'title'                => esc_html__( 'Sign Language (Prerecorded)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'A sign language interpretation must be provided for all prerecorded audio content in synchronized media, ensuring that users who are deaf and use sign language can fully understand the spoken content.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#sign-language-prerecorded',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/sign-language-prerecorded.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#sign-language-prerecorded',
		'tags'                 => [ 'video', 'audio', 'media', 'sign language', 'interpretation' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '1.2.7',
		'title'                => esc_html__( 'Extended Audio Description (Prerecorded)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For all prerecorded multimedia content, extended audio descriptions must be provided when necessary—meaning the video is paused to insert additional descriptions—so that users who are blind or have low vision can understand important visual details that can\'t be described during the natural pauses in the audio.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#extended-audio-description-prerecorded',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/extended-audio-description-prerecorded.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#extended-audio-description-prerecorded',
		'tags'                 => [ 'video', 'audio', 'media', 'audio description', 'description', 'extended' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '1.2.8',
		'title'                => esc_html__( 'Media Alternative (Prerecorded)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For prerecorded synchronized media, a full text alternative (such as a transcript) is provided that gives all the information in both the video and audio tracks, so users who cannot see or hear the content can still access it.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
			esc_html__( 'Deaf-blind people', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#media-alternative-prerecorded',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/media-alternative-prerecorded.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#media-alternative-prerecorded',
		'tags'                 => [ 'video', 'audio', 'media', 'transcript', 'text alternative' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '1.2.9',
		'title'                => esc_html__( 'Audio-only (Live)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For live audio-only content, a text alternative is provided in real time so people who cannot hear the audio can still access the information.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.2 Time-based Media', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#audio-only-live',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/audio-only-live.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#audio-only-live',
		'tags'                 => [ 'audio', 'media', 'live', 'transcript', 'text alternative' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '1.3.1',
		'title'                => esc_html__( 'Info and Relationships', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Information, structure, and relationships in content are programmatically determined or available in text, so assistive technologies can present them to users.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.3 Adaptable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with cognitive differences who view interfaces using an adapted layout', 'accessibility-checker' ),
			esc_html__( 'People with dexterity impairments using voice control', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#info-and-relationships',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/info-and-relationships.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#info-and-relationships',
		'tags'                 => [ 'structure', 'heading', 'list', 'table', 'form', 'label', 'aria', 'landmark', 'relationship', 'programmatic' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.3.2',
		'title'                => esc_html__( 'Meaningful Sequence', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Content is presented in a meaningful order so that reading or navigation order does not confuse users when using assistive technology.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.3 Adaptable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with cognitive differences who view interfaces using an adapted layout', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#meaningful-sequence',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/meaningful-sequence.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#meaningful-sequence',
		'tags'                 => [ 'order', 'reading', 'sequence', 'screen reader', 'programmatic', 'content' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.3.3',
		'title'                => esc_html__( 'Sensory Characteristics', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Instructions do not rely only on sensory characteristics such as shape, color, size, or sound, so everyone can understand them.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.3 Adaptable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#sensory-characteristics',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/sensory-characteristics.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#sensory-characteristics',
		'tags'                 => [ 'instruction', 'color', 'shape', 'sound', 'size', 'sensory', 'label' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.3.4',
		'title'                => esc_html__( 'Orientation', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Content does not restrict its view and operation to a single display orientation, such as portrait or landscape, unless a specific orientation is essential.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '1.3 Adaptable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#orientation',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/orientation.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#orientation',
		'tags'                 => [ 'orientation', 'mobile', 'portrait', 'landscape', 'responsive', 'layout' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.3.5',
		'title'                => esc_html__( 'Identify Input Purpose', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The purpose of input fields that collect information about the user can be programmatically determined, so browsers and assistive technologies can help users fill them in.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '1.3 Adaptable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with cognitive disabilities or limited short-term memory or reading disabilities', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#identify-input-purpose',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/identify-input-purpose.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#identify-input-purpose',
		'tags'                 => [ 'input', 'form', 'autocomplete', 'purpose', 'field', 'user info' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.3.6',
		'title'                => esc_html__( 'Identify Purpose', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The purpose of user interface components, icons, and regions can be programmatically determined, so assistive technologies can adapt the experience for users.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '1.3 Adaptable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with cognitive disabilities or limited short-term memory or reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#identify-purpose',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/identify-purpose.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#identify-purpose',
		'tags'                 => [ 'input', 'form', 'button', 'icon', 'region', 'aria', 'purpose', 'role' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '1.4.1',
		'title'                => esc_html__( 'Use of Color', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Color is not used as the only way to convey information, indicate an action, prompt a response, or distinguish a visual element.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
			esc_html__( 'Color blind people', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#use-of-color',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/use-of-color.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#use-of-color',
		'tags'                 => [ 'color', 'visual', 'link', 'button', 'chart', 'graph', 'distinguish' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.10',
		'title'                => esc_html__( 'Reflow', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Content can be viewed without scrolling in more than one direction, so users can easily read and interact with it on small screens or when zoomed in.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People on mobile devices', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#reflow',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/reflow.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#reflow',
		'tags'                 => [ 'responsive', 'zoom', 'mobile', 'layout', 'scroll', 'css', 'viewport' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.11',
		'title'                => esc_html__( 'Non-text Contrast', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Visual information needed to identify user interface components and graphical objects has a contrast ratio of at least 3:1 against adjacent colors.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#non-text-contrast',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/non-text-contrast.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#non-text-contrast',
		'tags'                 => [ 'contrast', 'button', 'icon', 'focus', 'visual', 'ui component', 'graphics' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.12',
		'title'                => esc_html__( 'Text Spacing', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Users can override text spacing to improve readability without losing content or functionality.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#text-spacing',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/text-spacing.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#text-spacing',
		'tags'                 => [ 'text', 'spacing', 'css', 'line height', 'letter spacing', 'word spacing', 'paragraph' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.13',
		'title'                => esc_html__( 'Content on Hover or Focus', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If extra content appears on hover or focus, users can dismiss it, move their pointer over it, and it remains visible until dismissed or focus is moved.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities', 'accessibility-checker' ),
			esc_html__( 'People with dexterity impairments', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#content-on-hover-or-focus',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/content-on-hover-or-focus.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#content-on-hover-or-focus',
		'tags'                 => [ 'hover', 'focus', 'tooltip', 'popup', 'menu', 'overlay', 'pointer', 'keyboard', 'dismiss' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.2',
		'title'                => esc_html__( 'Audio Control', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If audio plays automatically for more than 3 seconds, users must be able to pause, stop, or control the volume independently from the system volume.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
			esc_html__( 'People with ADD or difficulty focusing on content without distraction', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#audio-control',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/audio-control.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#audio-control',
		'tags'                 => [ 'audio', 'media', 'autoplay', 'volume', 'control', 'pause', 'stop', 'sound' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.3',
		'title'                => esc_html__( 'Contrast (Minimum)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Text and images of text must have a contrast ratio of at least 4.5:1, except for large text, incidental text, or logotypes.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People on mobile devices in a sunny environment', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#contrast-minimum',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/contrast-minimum.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#contrast-minimum',
		'tags'                 => [ 'contrast', 'text', 'color', 'visual', 'font', 'readability' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.4',
		'title'                => esc_html__( 'Resize text', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Text can be resized up to 200% without loss of content or functionality, so people with low vision can read it more easily.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#resize-text',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/resize-text.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#resize-text',
		'tags'                 => [ 'text', 'resize', 'zoom', 'font', 'css', 'responsive' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.5',
		'title'                => esc_html__( 'Images of Text', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If the same visual presentation can be made using text alone, do not use images of text except for decorative or essential purposes.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#images-of-text',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/images-of-text.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#images-of-text',
		'tags'                 => [ 'image', 'text', 'visual', 'font', 'decorative', 'essential', 'presentation' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '1.4.6',
		'title'                => esc_html__( 'Contrast (Enhanced)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Text and images of text must have a contrast ratio of at least 7:1, except for large text, incidental text, or logotypes.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People on mobile devices in a sunny environment', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#contrast-enhanced',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/contrast-enhanced.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#contrast-enhanced',
		'tags'                 => [ 'contrast', 'text', 'color', 'visual', 'font', 'readability', 'enhanced' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '1.4.7',
		'title'                => esc_html__( 'Low or No Background Audio', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For prerecorded audio-only content, background sounds are either low, can be turned off, or are not present, so speech is easier to understand.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Deaf and hard of hearing people', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#low-or-no-background-audio',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/low-or-no-background-audio.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#low-or-no-background-audio',
		'tags'                 => [ 'audio', 'media', 'background', 'sound', 'speech', 'clarity' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '1.4.8',
		'title'                => esc_html__( 'Visual Presentation', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Users can choose how blocks of text are presented, such as foreground and background colors, width, and line spacing, to make reading easier.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#visual-presentation',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/visual-presentation.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#visual-presentation',
		'tags'                 => [ 'visual', 'presentation', 'text', 'color', 'spacing', 'line length', 'css', 'readability' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '1.4.9',
		'title'                => esc_html__( 'Images of Text (No Exception)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Images of text are only used for pure decoration or where a particular presentation of text is essential, with no exceptions.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '1.4 Distinguishable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Perceivable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#images-of-text-no-exception',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/images-of-text-no-exception.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#images-of-text-no-exception',
		'tags'                 => [ 'image', 'text', 'visual', 'decorative', 'essential', 'presentation', 'no exception' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.1.1',
		'title'                => esc_html__( 'Keyboard', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'All functionality is available using a keyboard, so users who cannot use a mouse can still operate the content.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.1 Keyboard Accessible', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#keyboard',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/keyboard.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#keyboard',
		'tags'                 => [ 'keyboard', 'focus', 'tab', 'navigation', 'input', 'control', 'access' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.1.2',
		'title'                => esc_html__( 'No Keyboard Trap', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Keyboard users can move to and away from all parts of the content without getting stuck.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.1 Keyboard Accessible', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#no-keyboard-trap',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/no-keyboard-trap.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#no-keyboard-trap',
		'tags'                 => [ 'keyboard', 'focus', 'navigation', 'control', 'escape', 'tab' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.1.3',
		'title'                => esc_html__( 'Keyboard (No Exception)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'All functionality is available from a keyboard interface, with no exceptions.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.1 Keyboard Accessible', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Blind and low vision people', 'accessibility-checker' ),
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#keyboard-no-exception',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/keyboard-no-exception.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#keyboard-no-exception',
		'tags'                 => [ 'keyboard', 'focus', 'tab', 'navigation', 'input', 'control', 'access' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.1.4',
		'title'                => esc_html__( 'Character Key Shortcuts', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If a keyboard shortcut uses only letters, numbers, or punctuation, users can turn it off, change it, or it only works when the element has focus.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.1 Keyboard Accessible', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#character-key-shortcuts',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/character-key-shortcuts.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#character-key-shortcuts',
		'tags'                 => [ 'keyboard', 'shortcut', 'accesskey', 'focus', 'input', 'control', 'customization' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.2.1',
		'title'                => esc_html__( 'Timing Adjustable', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If a time limit is set, users can turn it off, adjust it, or extend it unless the time limit is essential.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.2 Enough Time', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who need more time to react', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#timing-adjustable',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/timing-adjustable.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#timing-adjustable',
		'tags'                 => [ 'timing', 'timeout', 'session', 'form', 'input', 'timer', 'adjustable', 'control' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.2.2',
		'title'                => esc_html__( 'Pause, Stop, Hide', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Users can pause, stop, or hide moving, blinking, scrolling, or auto-updating information that starts automatically and lasts more than five seconds.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.2 Enough Time', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with ADD or difficulty focusing on content without distraction', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#pause-stop-hide',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/pause-stop-hide.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#pause-stop-hide',
		'tags'                 => [ 'animation', 'pause', 'stop', 'hide', 'marquee', 'carousel', 'auto-update', 'scroll', 'blink', 'motion' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.2.3',
		'title'                => esc_html__( 'No Timing', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If an element requires a response in a specific time, users can turn off the timer, adjust the time limit, or the timing is not essential.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.2 Enough Time', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who need more time to react', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#no-timing',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/no-timing.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#no-timing',
		'tags'                 => [ 'timing', 'timeout', 'session', 'form', 'input', 'timer', 'no timing', 'control' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.2.4',
		'title'                => esc_html__( 'Interruptions', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Users are not interrupted by unexpected changes in content, and can control interruptions such as auto-updating content.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.2 Enough Time', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who need more time to react', 'accessibility-checker' ),
			esc_html__( 'People with ADD or difficulty focusing on content without distraction', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#interruptions',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/interruptions.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#interruptions',
		'tags'                 => [ 'interrupt', 'notification', 'auto-update', 'alert', 'timing', 'session', 'control' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.2.5',
		'title'                => esc_html__( 'Re-authenticating', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If users are logged out due to inactivity, they are notified before the session expires and can continue the session without losing data.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.2 Enough Time', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who need more time to react', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#re-authenticating',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/re-authenticating.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#re-authenticating',
		'tags'                 => [ 'authentication', 'session', 'timeout', 'login', 'form', 're-authentication', 'data loss' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.2.6',
		'title'                => esc_html__( 'Timeouts', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Users are warned of impending timeouts and can request more time or save their work before timing out.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.2 Enough Time', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who need more time to react', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#timeouts',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/timeouts.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#timeouts',
		'tags'                 => [ 'timeout', 'warning', 'session', 'timer', 'form', 'input', 'save', 'extend' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.3.1',
		'title'                => esc_html__( 'Three Flashes or Below Threshold', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Content does not flash more than three times in any one second period, reducing the risk of seizures for people with photosensitive epilepsy.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.3 Seizures and Physical Reactions', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with photosensitive epilepsy and other photosensitive seizure disorders', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#three-flashes-or-below-threshold',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/three-flashes-or-below-threshold.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#three-flashes-or-below-threshold',
		'tags'                 => [ 'animation', 'flash', 'video', 'media', 'seizure', 'motion', 'threshold' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.3.2',
		'title'                => esc_html__( 'Three Flashes', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'No part of the content flashes more than three times per second, with no exceptions.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.3 Seizures and Physical Reactions', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with photosensitive epilepsy and other photosensitive seizure disorders', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#three-flashes',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/three-flashes.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#three-flashes',
		'tags'                 => [ 'animation', 'flash', 'video', 'media', 'seizure', 'motion', 'no exception' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.3.3',
		'title'                => esc_html__( 'Animation from Interactions', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Users can turn off non-essential animations triggered by interactions, helping those who are sensitive to motion.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.3 Seizures and Physical Reactions', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with vestibular (inner ear) disorders where movement can cause dizziness/headaches and nausea', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#animation-from-interactions',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/animation-from-interactions.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#animation-from-interactions',
		'tags'                 => [ 'animation', 'motion', 'interaction', 'toggle', 'reduce motion', 'preference' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.4.1',
		'title'                => esc_html__( 'Bypass Blocks', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'A way is available to skip repeated blocks of content, such as navigation menus, so users can quickly reach the main content.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#bypass-blocks',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/bypass-blocks.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#bypass-blocks',
		'tags'                 => [ 'skip link', 'navigation', 'menu', 'landmark', 'main', 'header', 'footer', 'bypass', 'block' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.4.10',
		'title'                => esc_html__( 'Section Headings', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Section headings are used to organize content, making it easier to understand and navigate.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities or limited short-term memory or reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#section-headings',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/section-headings.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#section-headings',
		'tags'                 => [ 'heading', 'section', 'structure', 'navigation', 'outline', 'label' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.4.11',
		'title'                => esc_html__( 'Focus Not Obscured (Minimum)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'When an element receives keyboard focus, it is at least partially visible and not hidden by other content.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#focus-not-obscured-minimum',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/focus-not-obscured-minimum.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#focus-not-obscured-minimum',
		'tags'                 => [ 'focus', 'keyboard', 'visibility', 'tab', 'input', 'control', 'obscured' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.4.12',
		'title'                => esc_html__( 'Focus Not Obscured (Enhanced)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'When an element receives keyboard focus, it is fully visible and not hidden by other content.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#focus-not-obscured-enhanced',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/focus-not-obscured-enhanced.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#focus-not-obscured-enhanced',
		'tags'                 => [ 'focus', 'keyboard', 'visibility', 'tab', 'input', 'control', 'obscured', 'enhanced' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.4.13',
		'title'                => esc_html__( 'Focus Appearance', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The visible focus indicator for interactive elements is clearly visible and meets minimum size and contrast requirements.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#focus-appearance',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/focus-appearance.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#focus-appearance',
		'tags'                 => [ 'focus', 'keyboard', 'indicator', 'visible', 'contrast', 'size', 'appearance' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.4.2',
		'title'                => esc_html__( 'Page Titled', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Web pages have descriptive titles that help users understand the topic or purpose of the page.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with cognitive disabilities or limited short-term memory or reading disabilities', 'accessibility-checker' ),
			esc_html__( 'People with severe mobility impairments relying on audio to navigate the web', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#page-titled',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/page-titled.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#page-titled',
		'tags'                 => [ 'title', 'page', 'head', 'document', 'context', 'navigation' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.4.3',
		'title'                => esc_html__( 'Focus Order', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'When navigating with a keyboard, focus moves in a logical order that preserves meaning and operability.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#focus-order',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/focus-order.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#focus-order',
		'tags'                 => [ 'focus', 'order', 'keyboard', 'tab', 'navigation', 'sequence', 'logic' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.4.4',
		'title'                => esc_html__( 'Link Purpose (In Context)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The purpose of each link is clear from the link text and the context in which it appears, so users know where the link will take them.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#link-purpose-in-context',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/link-purpose-in-context.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#link-purpose-in-context',
		'tags'                 => [ 'link', 'purpose', 'context', 'anchor', 'navigation', 'text' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.4.5',
		'title'                => esc_html__( 'Multiple Ways', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'More than one way is available to locate a web page within a set of web pages, except where the web page is the result of a query.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#multiple-ways',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/multiple-ways.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#multiple-ways',
		'tags'                 => [ 'navigation', 'menu', 'search', 'sitemap', 'breadcrumbs', 'multiple ways', 'find' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.4.6',
		'title'                => esc_html__( 'Headings and Labels', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Headings and labels describe the topic or purpose of content, and are not ambiguous or misleading.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities or limited short-term memory or reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#headings-and-labels',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/headings-and-labels.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#headings-and-labels',
		'tags'                 => [ 'heading', 'label', 'form', 'input', 'aria-label', 'aria-labelledby', 'legend', 'fieldset', 'structure', 'search' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.4.7',
		'title'                => esc_html__( 'Focus Visible', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'It is visible when an element has keyboard focus, so users navigating with a keyboard can see which element is active.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#focus-visible',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/focus-visible.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#focus-visible',
		'tags'                 => [ 'focus', 'keyboard', 'tab', 'input', 'button', 'link', 'visible', 'outline', 'navigation' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.4.8',
		'title'                => esc_html__( 'Location', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Users are informed about their location within a set of web pages, such as through breadcrumb navigation or a clear page title.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities or limited short-term memory or reading disabilities', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#location',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/location.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#location',
		'tags'                 => [ 'breadcrumb', 'navigation', 'location', 'page', 'title', 'header', 'aria-current', 'search' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.4.9',
		'title'                => esc_html__( 'Link Purpose (Link Only)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The purpose of each link can be determined by the link text alone, without needing additional context.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '2.4 Navigable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#link-purpose-link-only',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/link-purpose-link-only.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#link-purpose-link-only',
		'tags'                 => [ 'link', 'anchor', 'a', 'href', 'button', 'purpose', 'text', 'search' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.5.1',
		'title'                => esc_html__( 'Pointer Gestures', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For functionality that can be operated by a pointer gesture, at least one of the following is true: the gesture can be performed by a simple tap, the gesture can be performed with a single finger, or the gesture can be performed with a voice command.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.5 Input Modalities', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#pointer-gestures',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/pointer-gestures.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#pointer-gestures',
		'tags'                 => [ 'pointer', 'gesture', 'touch', 'mouse', 'drag', 'swipe', 'tap', 'input', 'search' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.5.2',
		'title'                => esc_html__( 'Pointer Cancellation', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For actions that require a pointer gesture, the user can cancel the action by moving the pointer away before the action is completed.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.5 Input Modalities', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#pointer-cancellation',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/pointer-cancellation.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#pointer-cancellation',
		'tags'                 => [ 'pointer', 'gesture', 'touch', 'mouse', 'cancel', 'input', 'button', 'search' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.5.3',
		'title'                => esc_html__( 'Label in Name', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The name of user interface components includes text labels that describe their purpose, so assistive technologies can convey this information to users.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.5 Input Modalities', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with dexterity impairments using voice control', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#label-in-name',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/label-in-name.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#label-in-name',
		'tags'                 => [ 'label', 'name', 'button', 'input', 'aria-label', 'aria-labelledby', 'form', 'search' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.5.4',
		'title'                => esc_html__( 'Motion Actuation', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Functionality that can be operated by a motion gesture can also be operated by a simple touch or click, so users who cannot perform motion gestures can still use the functionality.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.5 Input Modalities', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#motion-actuation',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/motion-actuation.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#motion-actuation',
		'tags'                 => [ 'motion', 'gesture', 'device motion', 'touch', 'input', 'button', 'search' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '2.5.5',
		'title'                => esc_html__( 'Target Size', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The size of the target for pointer gestures is large enough to be easily activated, even by users with limited dexterity.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.5 Input Modalities', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People using a mobile device in environments where they are exposed to shaking (such as public transportation)', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments including hand tremors', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
			esc_html__( 'People with with large fingers or who are operating the device with only a part of their finger or knuckle', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#target-size',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/target-size.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#target-size',
		'tags'                 => [ 'target', 'size', 'button', 'link', 'input', 'touch', 'pointer', 'search' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.5.6',
		'title'                => esc_html__( 'Concurrent Input Mechanisms', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If an input mechanism requires a specific posture or movement, an alternative input mechanism is available that does not have the same requirement.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '2.5 Input Modalities', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#concurrent-input-mechanisms',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/concurrent-input-mechanisms.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#concurrent-input-mechanisms',
		'tags'                 => [ 'input', 'mechanism', 'keyboard', 'mouse', 'touch', 'voice', 'alternative', 'concurrent', 'search' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.5.7',
		'title'                => esc_html__( 'Dragging Movements', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For functionality that requires dragging movements, the action can also be completed by simpler actions, so users with limited dexterity can still use the functionality.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '2.5 Input Modalities', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#dragging-movements',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/dragging-movements.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#dragging-movements',
		'tags'                 => [ 'drag', 'drag-and-drop', 'mouse', 'touch', 'keyboard', 'input', 'movement', 'alternative', 'search' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '2.5.8',
		'title'                => esc_html__( 'Target Size (Minimum)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The target size for any interactive element is at least 44x44 pixels, so it can be easily activated by users with various abilities.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '2.5 Input Modalities', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Operable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People on mobile devices', 'accessibility-checker' ),
			esc_html__( 'People with dexterity impairments', 'accessibility-checker' ),
			esc_html__( 'People with low vision', 'accessibility-checker' ),
			esc_html__( 'People with with large fingers or who are operating the device with only a part of their finger or knuckle', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#target-size-minimum',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/target-size-minimum.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#target-size-minimum',
		'tags'                 => [ 'target', 'size', 'button', 'link', 'input', 'touch', 'pointer', 'minimum', 'search' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.1.1',
		'title'                => esc_html__( 'Language of Page', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The default language of each web page is programmatically determined, so assistive technologies can use the correct language settings.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.1 Readable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Language learners', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#language-of-page',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/language-of-page.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#language-of-page',
		'tags'                 => [ 'html', 'lang', 'language', 'page', 'document', 'assistive technology' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],

	],
	[
		'number'               => '3.1.2',
		'title'                => esc_html__( 'Language of Parts', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'The language of each part of the content can be programmatically determined, so assistive technologies can provide appropriate language support.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.1 Readable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Language learners', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#language-of-parts',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/language-of-parts.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#language-of-parts',
		'tags'                 => [ 'html', 'lang', 'span', 'language', 'parts', 'assistive technology' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],

	],
	[
		'number'               => '3.1.3',
		'title'                => esc_html__( 'Unusual Words', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Unusual or rare words are defined or explained, so users with cognitive disabilities or who are not familiar with the language can understand them.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.1 Readable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#unusual-words',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/unusual-words.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#unusual-words',
		'tags'                 => [ 'abbr', 'dfn', 'glossary', 'definition', 'tooltip', 'unusual word', 'explanation' ],
		'also_applies_to'      => [],

	],
	[
		'number'               => '3.1.4',
		'title'                => esc_html__( 'Abbreviations', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Abbreviations and acronyms are defined the first time they are used, so users can understand their meaning.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.1 Readable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#abbreviations',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/abbreviations.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#abbreviations',
		'tags'                 => [ 'abbr', 'acronym', 'dfn', 'glossary', 'definition', 'tooltip', 'explanation' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.1.5',
		'title'                => esc_html__( 'Reading Level', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Content is written in a clear and simple language, and does not require specialized knowledge to understand.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.1 Readable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#reading-level',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/reading-level.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#reading-level',
		'tags'                 => [ 'text', 'reading level', 'plain language', 'summary', 'explanation', 'content' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.1.6',
		'title'                => esc_html__( 'Pronunciation', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'A pronunciation guide is available for unusual or difficult words, so users can understand how to say them.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.1 Readable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'Language learners', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#pronunciation',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/pronunciation.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#pronunciation',
		'tags'                 => [ 'pronunciation', 'phonetic', 'tooltip', 'explanation', 'assistive technology' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.2.1',
		'title'                => esc_html__( 'On Focus', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'When an element receives focus, it does not cause unexpected changes that could confuse the user, and it is clear how to interact with the element.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.2 Predictable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#on-focus',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/on-focus.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#on-input',
		'tags'                 => [ 'focus', 'input', 'button', 'link', 'form', 'change', 'interaction', 'event' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],

	],
	[
		'number'               => '3.2.2',
		'title'                => esc_html__( 'On Input', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'When an input field receives data, it does not cause unexpected changes that could confuse the user, and it is clear how the data will be used.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.2 Predictable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#on-input',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/on-input.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#on-input',
		'tags'                 => [ 'input', 'form', 'change', 'event', 'submit', 'interaction', 'field' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],

	],
	[
		'number'               => '3.2.3',
		'title'                => esc_html__( 'Consistent Navigation', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Navigation mechanisms are consistent across the website, so users can predictably move through the site.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.2 Predictable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#consistent-navigation',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/consistent-navigation.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#consistent-navigation',
		'tags'                 => [ 'navigation', 'menu', 'link', 'header', 'footer', 'sidebar', 'consistency' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software) - does not apply to non-web software',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs) - does not apply to non-web docs',
			],
		],
	],
	[
		'number'               => '3.2.4',
		'title'                => esc_html__( 'Consistent Identification', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Components that have the same functionality within a set of web pages are identified consistently, so users can recognize them easily.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.2 Predictable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#consistent-identification',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/consistent-identification.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#consistent-identification',
		'tags'                 => [ 'button', 'link', 'icon', 'input', 'form', 'label', 'consistency', 'identification' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software) - does not apply to non-web software',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs) - does not apply to non-web docs',
			],
		],
	],
	[
		'number'               => '3.2.5',
		'title'                => esc_html__( 'Change on Request', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Changes in content or context that occur as a result of user input do not occur unexpectedly, and users are notified of any changes.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.2 Predictable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People who rely on keyboard control', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'People with dexterity and mobility impairments', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#change-on-request',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/change-on-request.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#change-on-request',
		'tags'                 => [ 'input', 'form', 'button', 'link', 'change', 'event', 'notification', 'context', 'request' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.2.6',
		'title'                => esc_html__( 'Consistent Help', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Help and documentation are available and are easy to find, so users can get assistance when needed.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '3.2 Predictable', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#consistent-help',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/consistent-help.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#consistent-help',
		'tags'                 => [ 'help', 'support', 'documentation', 'contact', 'faq', 'search', 'assistance' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.3.1',
		'title'                => esc_html__( 'Error Identification', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If an input error is detected and suggestions for correction are known, then the suggestions are provided to the user.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#error-identification',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/error-identification.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#error-identification',
		'tags'                 => [ 'error', 'input', 'form', 'validation', 'message', 'field', 'suggestion', 'notification' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '3.3.2',
		'title'                => esc_html__( 'Labels or Instructions', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Labels or instructions are provided when an input field requires user input, so users know what information is needed.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#labels-or-instructions',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/labels-or-instructions.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#labels-or-instructions',
		'tags'                 => [ 'label', 'input', 'form', 'instruction', 'field', 'aria-label', 'aria-labelledby', 'placeholder' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '3.3.3',
		'title'                => esc_html__( 'Error Suggestion', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'If an input error is detected and suggestions for correction are known, then the suggestions are provided to the user.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#error-suggestion',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/error-suggestion.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#error-suggestion',
		'tags'                 => [ 'error', 'input', 'form', 'validation', 'suggestion', 'message', 'field', 'notification' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '3.3.4',
		'title'                => esc_html__( 'Error Prevention (Legal, Financial, Data)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For web pages that require user input, users are warned of errors and can correct them before finalizing the submission.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#error-prevention-legal-financial-data',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/error-prevention-legal-financial-data.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#minimize-error-reversible',
		'tags'                 => [ 'error', 'input', 'form', 'validation', 'suggestion', 'message', 'field', 'notification' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '3.3.5',
		'title'                => esc_html__( 'Help', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Help and documentation are available and are easy to find, so users can get assistance when needed.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#help',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/help.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#help',
		'tags'                 => [ 'help', 'support', 'documentation', 'contact', 'faq', 'search', 'assistance' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.3.6',
		'title'                => esc_html__( 'Error Prevention (All)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'For web pages that require user input, users are warned of errors and can correct them before finalizing the submission.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with cognitive disabilities and reading disabilities', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#error-prevention-all',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/error-prevention-all.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#error-prevention-all',
		'tags'                 => [ 'error', 'input', 'form', 'validation', 'prevention', 'confirmation', 'review', 'submission', 'data' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.3.7',
		'title'                => esc_html__( 'Redundant Entry', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Users are not required to enter the same information multiple times, reducing the risk of errors and saving time.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'All people including those who do not identify as disabled', 'accessibility-checker' ),
			esc_html__( 'People with severe mobility impairments relying on audio to navigate the web', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#redundant-entry',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/redundant-entry.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#redundant-entry',
		'tags'                 => [ 'input', 'form', 'redundant', 'entry', 'field', 'auto-fill', 'user info' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.3.8',
		'title'                => esc_html__( 'Accessible Authentication (Minimum)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Authentication processes do not rely on cognitive function tests like remembering a password or solving a puzzle, unless an alternative is provided.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with cognitive disabilities or limited short-term memory or reading disabilities', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#accessible-authentication-minimum',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/accessible-authentication-minimum.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#accessible-authentication-minimum',
		'tags'                 => [ 'authentication', 'login', 'password', 'captcha', 'alternative', 'cognitive', 'form', 'user' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '3.3.9',
		'title'                => esc_html__( 'Accessible Authentication (Enhanced)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Enhanced authentication methods are provided that do not rely on cognitive function tests, such as remembering passwords, and do not require a test of cognitive function.', 'accessibility-checker' ),
		'level'                => 'AAA',
		'version'              => '2.2',
		'guidelines'           => esc_html__( '3.3 Input Assistance', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Understandable', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with cognitive disabilities or limited short-term memory or reading disabilities', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG22/quickref/#accessible-authentication-enhanced',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG22/Understanding/accessible-authentication-enhanced.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG22/#accessible-authentication-enhanced',
		'tags'                 => [ 'authentication', 'login', 'password', 'captcha', 'alternative', 'cognitive', 'form', 'user', 'enhanced' ],
		'also_applies_to'      => [],
	],
	[
		'number'               => '4.1.1',
		'title'                => esc_html__( 'Parsing', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Content is coded so that assistive technologies can accurately interpret and present it to users.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '4.1 Compatible', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Robust', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with dexterity impairments using voice control', 'accessibility-checker' ),
			esc_html__( 'People with low vision using screen magnifiers', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#parsing',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/parsing.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#parsing',
		'tags'                 => [ 'html', 'xml', 'syntax', 'element', 'attribute', 'aria', 'parsing', 'structure', 'assistive technology' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.1.1.1.2 (Closed Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '4.1.2',
		'title'                => esc_html__( 'Name, Role, Value', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'User interface components have names, roles, and values that can be programmatically determined, so assistive technologies can communicate them to users.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '2.0',
		'guidelines'           => esc_html__( '4.1 Compatible', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Robust', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'People with dexterity impairments using voice control', 'accessibility-checker' ),
			esc_html__( 'People with low vision using screen magnifiers', 'accessibility-checker' ),
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#name-role-value',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/name-role-value.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#name-role-value',
		'tags'                 => [ 'aria', 'role', 'name', 'value', 'button', 'input', 'form', 'widget', 'state', 'property', 'assistive technology' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
			'Revised Section 508' => [
				'501 (Web)(Software)',
				'504.2 (Authoring Tool)',
				'602.3 (Support Docs)',
			],
		],
	],
	[
		'number'               => '4.1.3',
		'title'                => esc_html__( 'Status Messages', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Status messages are programmatically determined so that assistive technologies can notify users of important changes, such as form errors or updates.', 'accessibility-checker' ),
		'level'                => 'AA',
		'version'              => '2.1',
		'guidelines'           => esc_html__( '4.1 Compatible', 'accessibility-checker' ),
		'principles'           => esc_html__( 'Robust', 'accessibility-checker' ),
		'impacted_populations' => [
			esc_html__( 'Screen reader users (blind/low vision and neurodivergent people)', 'accessibility-checker' ),
		],
		'how_to_meet_url'      => 'https://www.w3.org/WAI/WCAG21/quickref/#status-messages',
		'understanding_url'    => 'https://www.w3.org/WAI/WCAG21/Understanding/status-messages.html',
		'wcag_url'             => 'https://www.w3.org/TR/WCAG21/#status-messages',
		'tags'                 => [ 'status', 'aria-live', 'alert', 'notification', 'form', 'update', 'assistive technology', 'message' ],
		'also_applies_to'      => [
			'EN 301 549 Criteria' => [
				'9.1.1.1 (Web)',
				'10.1.1.1 (Non-web document)',
				'11.1.1.1.1 (Open Functionality Software)',
				'11.8.2 (Authoring Tool)',
				'12.1.2 (Product Docs)',
				'12.2.4 (Support Docs)',
			],
		],
	],
	[
		'number'               => '0.1',
		'title'                => esc_html__( 'Best Practice', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'A usability improvement or convention that enhances accessibility but is not required for WCAG compliance.', 'accessibility-checker' ),
		'level'                => 'Best Practice',
		'version'              => '',
		'guidelines'           => '',
		'principles'           => '',
		'impacted_populations' => [],
		'how_to_meet_url'      => '',
		'understanding_url'    => '',
		'wcag_url'             => '',
		'tags'                 => [],
		'also_applies_to'      => [],
	],
	[
		'number'               => '0.2',
		'title'                => esc_html__( 'Non-WCAG Accessibility Issue', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'A problem that affects users with disabilities but doesn\'t map directly to a current WCAG success criterion.', 'accessibility-checker' ),
		'level'                => 'Best Practice',
		'version'              => '',
		'guidelines'           => '',
		'principles'           => '',
		'impacted_populations' => [],
		'how_to_meet_url'      => '',
		'understanding_url'    => '',
		'wcag_url'             => '',
		'tags'                 => [],
		'also_applies_to'      => [],
	],
	[
		'number'               => '0.3',
		'title'                => esc_html__( 'Manual Testing Needed (Multiple Success Criteria)', 'accessibility-checker' ),
		'criteria_description' => esc_html__( 'Used for documents, media, and other non-scannable content that require manual testing and are likely to fail multiple WCAG success criteria.', 'accessibility-checker' ),
		'level'                => 'A',
		'version'              => '',
		'guidelines'           => '',
		'principles'           => '',
		'impacted_populations' => [],
		'how_to_meet_url'      => '',
		'understanding_url'    => '',
		'wcag_url'             => '',
		'tags'                 => [],
		'also_applies_to'      => [],
	],
];
