<?php
/**
 * Class for the customizer importer.
 *
 * Code is mostly from the Customizer Export/Import plugin.
 *
 * @see https://wordpress.org/plugins/customizer-export-import/
 * @package Ablocks
 */

namespace ABlocks\import;

use ABlocks\Helper;
use stdClass;
use WP_Error;

class CustomizerImporter {

	/**
	 * Import customizer from a DAT file, generated by the Customizer Export/Import plugin.
	 *
	 * @param string $customizer_import_file_path path to the customizer import file.
	 */
	public static function import( string $customizer_import_file_path ) {

		// Try to import the customizer settings.
		$results = self::import_customizer_options( $customizer_import_file_path );

		// Check for errors, else write the results to the log file.
		if ( is_wp_error( $results ) ) {
			$error_message = $results->get_error_message();
			Helper::emit_sse_message([
				'action' => 'log',
				'level' => 'warning',
				'message' => $error_message,
			]);
			Helper::emit_sse_message([
				'action' => 'log',
				'level' => 'warning',
				'message' => __( 'Failed to import Customizer options.', 'ablocks' ),
			]);
		} else {
			Helper::emit_sse_message([
				'action' => 'log',
				'level' => 'info',
				'message' => __( 'Customizer settings import finished!', 'ablocks' ),
			]);
		}//end if

		// Delete the temporary file if it exists using wp_delete_file.
		if ( file_exists( $customizer_import_file_path ) && dirname( $customizer_import_file_path ) === sys_get_temp_dir() ) {
			wp_delete_file( $customizer_import_file_path );
		}
	}


	/**
	 * Imports uploaded mods and calls WordPress core customize_save actions so
	 * themes that hook into them can act before mods are saved to the database.
	 *
	 * Update: WP core customize_save actions were removed, because of some errors.
	 *
	 * @param string $import_file_path Path to the import file.
	 *
	 * @return void|WP_Error
	 */
	public static function import_customizer_options( string $import_file_path ) {
		// Setup global vars.
		global $wp_customize;

		// Setup internal vars.
		$template = get_template();

		// Make sure we have an import file.
		if ( ! file_exists( $import_file_path ) ) {
			return new WP_Error(
				'missing_customizer_import_file',
				sprintf( /* translators: %s - file path */
					esc_html__( 'Error: The customizer import file is missing! File path: %s', 'ablocks' ),
					$import_file_path
				)
			);
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
		$raw = file_get_contents( $import_file_path );

		// Make sure we got the data.
		if ( ! $raw ) {
			return new WP_Error(
				'missing_customizer_import_file',
				sprintf(
					// translators: %s is file path.
					__( "Can't read raw file '%s'!", 'ablocks' ),
					$import_file_path
				)
			);
		}

		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.serialize_unserialize
		$data = unserialize( $raw, array( 'allowed_classes' => false ) );

		// Data checks.
		if ( ! is_array( $data ) && ( ! isset( $data['template'] ) || ! isset( $data['mods'] ) ) ) {
			return new WP_Error(
				'customizer_import_data_error',
				esc_html__( 'Error: The customizer import file is not in a correct format. Please make sure to use the correct customizer import file.', 'ablocks' )
			);
		}
		if ( $data['template'] !== $template ) {
			return new WP_Error(
				'customizer_import_wrong_theme',
				esc_html__( 'Error: The customizer import file is not suitable for current theme. You can only import customizer settings for the same theme or a child theme.', 'ablocks' )
			);
		}

		// Import images.
		if ( apply_filters( 'ablocks/customizer_import_images', true ) ) {
			$data['mods'] = self::import_customizer_images( $data['mods'] );
		}

		// Import custom options.
		if ( isset( $data['options'] ) ) {
			// Require modified customizer options class.
			if ( ! class_exists( '\WP_Customize_Setting' ) ) {
				require_once ABSPATH . 'wp-includes/class-wp-customize-setting.php';
			}

			foreach ( $data['options'] as $option_key => $option_value ) {
				$option = new CustomizerOption( $wp_customize, $option_key, array(
					'default'    => '',
					'type'       => 'option',
					'capability' => 'edit_theme_options',
				) );

				$option->import( $option_value );
			}
		}

		// Should the customizer import use the WP customize_save* hooks?
		$use_wp_customize_save_hooks = apply_filters( 'ablocks/enable_wp_customize_save_hooks', false );

		if ( $use_wp_customize_save_hooks ) {
			do_action( 'customize_save', $wp_customize );
		}

		// Loop through the mods and save the mods.
		foreach ( $data['mods'] as $key => $val ) {
			if ( $use_wp_customize_save_hooks ) {
				do_action( 'customize_save_' . $key, $wp_customize );
			}

			set_theme_mod( $key, $val );
		}

		if ( $use_wp_customize_save_hooks ) {
			do_action( 'customize_save_after', $wp_customize );
		}
	}

	/**
	 * Helper function: Customizer import - imports images for settings saved as mods.
	 *
	 * @param array $mods An array of customizer mods.
	 *
	 * @return array The mods array with any new import data.
	 */
	private static function import_customizer_images( array $mods ): array {
		foreach ( $mods as $key => $val ) {
			if ( self::customizer_is_image_url( $val ) ) {
				$data = self::customizer_sideload_image( $val );
				if ( ! is_wp_error( $data ) ) {
					$mods[ $key ] = $data->url;

					// Handle header image controls.
					if ( isset( $mods[ $key . '_data' ] ) ) {
						$mods[ $key . '_data' ] = $data;
						update_post_meta( $data->attachment_id, '_wp_attachment_is_custom_header', get_stylesheet() );
					}
				}
			}
		}

		return $mods;
	}

	/**
	 * Helper function: Customizer import
	 * Taken from the core media_sideload_image function and
	 * modified to return an array of data instead of html.
	 *
	 * @param string $file The image file path.
	 */
	private static function customizer_sideload_image( string $file ) {
		$data = new stdClass();

		if ( ! function_exists( 'media_handle_sideload' ) ) {
			require_once ABSPATH . 'wp-admin/includes/media.php';
			require_once ABSPATH . 'wp-admin/includes/file.php';
			require_once ABSPATH . 'wp-admin/includes/image.php';
		}
		if ( ! empty( $file ) ) {
			// Set variables for storage, fix file filename for query strings.
			preg_match( '/[^\?]+\.(jpe?g|jpe|gif|png)\b/i', $file, $matches );
			$file_array = array();
			$file_array['name'] = basename( $matches[0] );

			// Download file to temp location.
			$file_array['tmp_name'] = download_url( $file );

			// If error storing temporarily, return the error.
			if ( is_wp_error( $file_array['tmp_name'] ) ) {
				return $file_array['tmp_name'];
			}

			// Do the validation and storage stuff.
			$id = media_handle_sideload( $file_array );

			// If error storing permanently, unlink.
			if ( is_wp_error( $id ) ) {
				unlink( $file_array['tmp_name'] );
				return $id;
			}

			// Build the object to return.
			$meta                = wp_get_attachment_metadata( $id );
			$data->attachment_id = $id;
			$data->url           = wp_get_attachment_url( $id );
			$data->thumbnail_url = wp_get_attachment_thumb_url( $id );
			$data->height        = $meta['height'];
			$data->width         = $meta['width'];
		}//end if

		return $data;
	}

	/**
	 * Checks to see whether a string is an image url or not.
	 *
	 * @param mixed $string The string to check.
	 *
	 * @return bool Whether the string is an image url or not.
	 */
	private static function customizer_is_image_url( $string ): bool {
		if ( ! is_string( $string ) ) {
			return false;
		}
		return preg_match( '/\.(jpg|jpeg|png|gif)/i', $string );
	}
}
